<?php

namespace IZON\DB\Definitions;

use \Exception;

use IZON\Logs\Logger;

/**
 * korenova definice, ktera slouzi jako zaklad pro vsechny definice
 */
abstract class AbstractDefinition {
    
    /**
     * jak se ma nazyvat sloupecek pro property 
     * @var string 
     */
    protected $columnName = NULL;
    
    /**
     * jestli muze byt null
     * @var boolean
     */
    protected $notNull = false;

    /**
     * ostatni parametry, krome columnName a notNull,
     * ktere se extrahuji z definicniho pole
     * @var array 
     */
    protected $parameters = [];


    /**
     * 
     * @var Logger 
     */
    private $log = NULL;
    
    /**
     * 
     * @param array $definitions
     * @throws \Exception
     */
    public function __construct(array $definitions) {
        $this->columnName = $this->extractDefinition("columnName", $definitions);
        $this->notNull = $this->extractDefinition("notNull", $definitions);
        
        // zkontroluje parametry
        $params = [];
        $suported = $this->getSupportedParameters();
        foreach($definitions as $key => $value) {
            if( in_array($key, $suported) ) {
                $params[$key] = $value;
                unset($definitions[$key]);
            }
        }
        $this->parameters = $params;
        
//        var_dump($params);
        
        if( !empty($definitions) ) {
//            var_dump($definitions);
            throw new \Exception("Definitions ". implode(", ", $definitions) ." not supported");
        }
    }
    
    public function getColumnName() {
        return $this->columnName;
    }
    
    public function getNotNull() {
        return $this->notNull;
    }
    
    /**
     * @return array array of deffinitions
     */
    public function getParameters() {
        return $this->parameters;
    }


    /**
     * extracts definition from array (returns value, unsets value in array), returns NULL if not
     * @param string $definitionName name of definition to extract
     * @param array $definitions definitions array
     * @param boolean $raiseExceprionIfNotPresent raise not present exception 
     * @param mixed $default default value to extract
     * @return mixed|NULL extracted definition or $default if not present
     */
    protected function extractDefinition($definitionName, 
                                            array &$definitions, // pass by reference to unset work properly
                                            $raiseExceprionIfNotPresent = false,
                                            $default = NULL) {
        $retval = $default;
        if( isset($definitions[$definitionName]) ) {
            $retval = $definitions[$definitionName];
            unset($definitions[$definitionName]);
        } else if( $raiseExceprionIfNotPresent ) {
            $e = new Exception("Definition $definitionName not set");
            $this->getLogger()->error("Definition $definitionName not set", ['exception' => $e]);
            throw $e;
        }
        return $retval;
    }
    
    /**
     * vraci podporovane parametry pro danou definici
     */
    protected abstract function getSupportedParameters();

    protected function getLogger() {
        if($this->log == null) {
            $this->log = Logger::getLogger(self::class);
        }
        return $this->log;
    }
    
    function __toString() {
        // TODO: nahradit necim co vypise jak klice tak hodnoty
        return "[". get_class($this) .", ". implode(", ", array_map(function($param) { return print_r($param, true); }, (array)$this)) ."]";
    }
}
