<?php

namespace IZON\DB\Impl;

use \Exception;

use \PDO;
use \PDOStatement;

use \IZON\Utils\Date;

use \IZON\IO\File;
use \IZON\IO\Image;

use \IZON\Logs\Logger;

use IZON\DB\DBConnection;
use IZON\DB\Utils\ClassDescription;
use IZON\DB\Definitions\AbstractDefinition;
use IZON\DB\Utils\AbstractPropertyDescription;
use IZON\DB\Utils\PropertyDescriptions\CharDescription;
use \IZON\DB\Definitions\TransientDefinition;

/**
 * MySQL db connector
 */
class DBConnectionMySQLImpl extends DBConnectionCommon implements DBConnection {
    
    /**
     * name of DB
     */
    const DB_NAME = "MySQL";
    
    private $dbName;
    
    /**
     * @var Logger 
     */
    private $log = NULL;
    
    function __construct($host, $dbName, $login, $password, 
                        $accessabeFileDir, $hiddenFilesDir,
                        array $settings = ["codepage" => 'utf8']) {
        $completeSettings = array_merge($this->getDefaultSettings(), $settings);
        $completeSettings['accessabeFileDir'] = $accessabeFileDir;
        $completeSettings['hiddenFilesDir'] = $hiddenFilesDir;
        
        // FIXME: kontrola, jestli se podarilo pripojit
        $pdo = new PDO("mysql:host=$host;dbname=$dbName;charset=". $completeSettings["codepage"], $login, $password);
        // neemuluj prepared statements pouzitelne od mysql 5.1.17 
        // http://stackoverflow.com/questions/10113562/pdo-mysql-use-pdoattr-emulate-prepares-or-not
        $pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);
        // nastavuje hlaseni chyb nayjimky vyjimky
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        // nekonvertovat prazdne stringy a nully a naopak pri fetch neni jen pro oracle
        $pdo->setAttribute(PDO::ATTR_ORACLE_NULLS, PDO::NULL_NATURAL);
        
        $this->dbName = $dbName;
        
        $this->log = Logger::getLogger(self::class);
        
        parent::__construct($pdo, self::DB_NAME, $completeSettings);
    }

    
    protected function createInsertSQLString(ClassDescription $classDescription) {
        $columnNames = '';
        $properties = '';
        foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            if( $key != $classDescription->getIdPropertyName()) { // updatuj vsechno krome id
                if($properties != '') {
                    $columnNames .= ', ';
                    $properties .= ', ';
                }
                $columnNames .= $value;
                $properties .= ":". $key;

            }
        }
        $sql = 'insert into '. $classDescription->getTableName() .' ('. $columnNames .') values ('. $properties .')';
        
        return $sql;
    }
    
    /**
     * vraci vygenerovane id posledniho vlozene zaznamu
     * @param PDO $pdo pdo db pripojeni
     * @param PDOStatement $statement statement ktery byl vykonan 
     * @return id
     */
    protected function getLastInsertedId(PDO $pdo, PDOStatement $statement) {
        return $pdo->lastInsertId();
    }
    
    /**
     * 
     * @param ClassDescription $classDescription
     * @param array $propertyDefinitions
     * @param array $propertyTypes
     * @return 
     */
    public function getPropertyDescriptions(ClassDescription $classDescription, $propertyDefinitions, $propertyTypes) {
        $tableName = $classDescription->getTableName();
        
        $statement = $this->getPDO()->prepare("select * from information_schema.columns 
                            where table_schema = :dbName and table_name = :tableName");
        $statement->bindValue(":dbName", $this->dbName);
        $statement->bindValue(":tableName", $tableName);
        $statement->execute();
        
        if($statement->rowCount() == 0) { // kontrola jestli 
            throw new Exception("Pro třídu ". $classDescription->getClassName() ." neexistuje tabulka ". $tableName);
        }
        
        $columnDescriptions = [];
        while( $row = $statement->fetch(PDO::FETCH_ASSOC) ) {
            $columnDescriptions[$row["COLUMN_NAME"]] = $row;
        }
        
        $statement->closeCursor();
        $statement = null;
        
        $propertyDescriptions = [];
        foreach( $propertyDefinitions as $key => $value ) {
            
            $propertyDefinition = $value;
            
            // transient se nemapuji
            if($propertyDefinition instanceof TransientDefinition) {
                
                continue;
            }
            
//            $dataType = $propertyTypes[$key];
            $columnName = NULL;
            if( $propertyDefinition != NULL ) {
                $columnName = $propertyDefinition->getColumnName();
            }
            if( $columnName == NULL ) { // column doesn't have defined column name
                $columnName = $this->namingStrategy->propertyToColumnName($key);
            }

            
            // init to null
            $propertyDescription = NULL;
            if( $propertyDefinition !== NULL ) { // initialize from PropertyDefinition
                $propertyDescription = $this->getPropertyDescriptionByProperityDefinition($classDescription, $propertyDefinition, $key, $columnName);
            }
            
            if( !isset($columnDescriptions[$columnName]) ) {
                throw new Exception('Pro property '.$key.' v tabulce '. $tableName .' neexistuje sploupec '.$columnName);
            }
            $columnDescription = $columnDescriptions[$columnName];
            
            if($propertyDescription == NULL) {
                $notNull = $columnDescription["IS_NULLABLE"] != "YES";
                $propertyDescription = $this->getPropertyDescriptionByDataType($classDescription,
                                                                                $propertyTypes[$key], 
                                                                                $key, 
                                                                                $columnName,
                                                                                $columnDescription["DATA_TYPE"],
                                                                                $notNull);
            }
            if($propertyDescription == NULL) { // nastavit defaultni na char
                $propertyDescription = new CharDescription($classDescription,
                                                            $key, 
                                                            $columnName,
                                                            $key,
                                                            $columnDescription["IS_NULLABLE"] != "YES");
            }
            
            // TODO: kontrola jestli odpovida informaci ziskanym z db
            
            $propertyDescriptions[$key] = $propertyDescription;
        }
        
        return $propertyDescriptions;
    }
    
    /**
     * 
     * @param AbstractDefinition $propertyDefinition
     * @param string $propertyName
     * @param string $columnName
     * @return AbstractPropertyDescription
     */
    protected function getPropertyDescriptionByProperityDefinition(ClassDescription $classDescription,
                                                                    AbstractDefinition $propertyDefinition, 
                                                                    $propertyName, 
                                                                    $columnName) {
        $columnReturnName = $propertyName;
        // TODO: predelat aby se pouzivaly PropertyDescription::isApplicable();
        switch( get_class($propertyDefinition) ) {
            case \IZON\DB\Definitions\CharDefinition::class;
                $def = new \IZON\DB\Utils\PropertyDescriptions\CharDescription($classDescription, 
                                                                                $propertyName, 
                                                                                $columnName, 
                                                                                $columnReturnName, 
                                                                                $propertyDefinition->getNotNull(),
                                                                                $propertyDefinition->getParameters());
                return $def;
            case \IZON\DB\Definitions\DateDefinition::class:
                return new \IZON\DB\Utils\PropertyDescriptions\DateDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case \IZON\DB\Definitions\TimeDefinition::class:
                return new \IZON\DB\Utils\PropertyDescriptions\TimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case \IZON\DB\Definitions\DateTimeDefinition::class:
                return new \IZON\DB\Utils\PropertyDescriptions\DateTimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case \IZON\DB\Definitions\IntegerDefinition::class:
                return new \IZON\DB\Utils\PropertyDescriptions\IntegerDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case \IZON\DB\Definitions\FloatDefinition::class:
                return new \IZON\DB\Utils\PropertyDescriptions\FloatDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case \IZON\DB\Definitions\CharDefinition::class:
                return new \IZON\DB\Utils\PropertyDescriptions\CharDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case \IZON\DB\Definitions\TextDefinition::class:
                return new \IZON\DB\Utils\PropertyDescriptions\TextDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
                case \IZON\DB\Definitions\FileDefinition::class:
                $params = $propertyDefinition->getParameters();
                $params["accessabeFileDir"] = $this->accessabeFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new \IZON\DB\Utils\PropertyDescriptions\FileDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
            case \IZON\DB\Definitions\ImageDefinition::class:
                $params = $propertyDefinition->getParameters();
                $params["accessabeFileDir"] = $this->accessabeFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new \IZON\DB\Utils\PropertyDescriptions\ImageDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
            
            default:
                return NULL;
        }
    }
    
    protected function getPropertyDescriptionByDataType(ClassDescription $classDescription,
                                                        $propertyType, $propertyName, 
                                                        $columnName, $columnDataType, $notNull) {
        $columnReturnName = $propertyName;
//        echo "$propertyType $propertyName $columnName\n";
        switch( $propertyType ) {
            case Date::class: // pro date
                switch( $columnDataType ) {
                    case "date":
                    case "year":
                        return new \IZON\DB\Utils\PropertyDescriptions\DateDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
                    case "time":
                        return new \IZON\DB\Utils\PropertyDescriptions\TimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
                    case "datetime":
                    case "timestamp":
                        return new \IZON\DB\Utils\PropertyDescriptions\DateTimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
                    default:
                        throw new Exception("Data type $columnDataType not supported.");
                }
            case File::class:
                $params = [];
                $params["accessabeFileDir"] = $this->accessabeFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new \IZON\DB\Utils\PropertyDescriptions\FileDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, $params);
            case Image::class:
                $params = [];
                $params["accessabeFileDir"] = $this->accessabeFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new \IZON\DB\Utils\PropertyDescriptions\ImageDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, $params);
            case "integer":
            case "int":
                return new \IZON\DB\Utils\PropertyDescriptions\IntegerDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
            case "float":
            case "double":
                return new \IZON\DB\Utils\PropertyDescriptions\FloatDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
            case "boolean":
                return new \IZON\DB\Utils\PropertyDescriptions\BooleanDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
            case "string":
                switch( $columnDataType ) {
                    case "text":
                        return new \IZON\DB\Utils\PropertyDescriptions\TextDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
                    default:
                        return new \IZON\DB\Utils\PropertyDescriptions\CharDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull);
                }
            default:
                return NULL;
        }
    }
}
