<?php

namespace IZON\DB\Impl;

use \Exception;
use \PDO;
use \PDOStatement;

use \IZON\Logs\Logger;

use \IZON\IO\File;

use IZON\DB\DBConnection;

use IZON\DB\NamingStrategy;


use IZON\DB\QueryHelpers;
use IZON\DB\QueryResult;
use IZON\DB\Utils\ClassDescription;
use IZON\DB\Utils\SqlStatementCache;
use IZON\DB\QueryFactory;

use IZON\DB\Paginator\PaginatorFactory;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\DB\Paginator\PageContent;

/**
 * Obsahuje slopecnou logiku sdilenou mezi vsemi dbPripojenumi
 */
abstract class DBConnectionCommon implements DBConnection {
    
    /**
     * pdo rozhrani pro pristup k db
     * @var PDO
     */
    protected $pdo;
    
    /**
     * jmeno pouzite db pro rozpoznani, ktere dotazy se maji pouzit
     * @var string 
     */
    protected $dbType;
    
    /**
     * jak se maji prevadet jmena trid na tabulky a jmena properit an tabulky
     * @var NamingStrategy 
     */
    protected $namingStrategy;
    
    /**
     * obsahuje popisy jednotlivych trid
     * @var array 
     */
    protected $classDescriptions = [];
    
    /**
     * cashe pro statementy
     * @var array SqlStatementCache 
     */
    protected $sqlStatementsCache = [];
    
    /**
     * cache jiz natazenych db objektu
     * @var array 
     */
    protected $objectCache = [];
    
    /**
     * v jakem adresari se maji ukladat soubory spravovane db vrstvou
     * @var string 
     */
    protected $accessabeFileDir = NULL;
    
    /**
     * @var string, v jakem adresari se nachazeji soubory, ktere nemaji byt pristupne primo pres web 
     */
    protected $hiddenFilesDir = NULL;

    /**
     * @var string do jakeho adresare se maji cachovat vygenerovane implementade dai
     */
    protected $daoImplCacheDir = NULL;

    /**
     * @var string jak se ma jmenovat obrazek, co se nahradi pokud obrazk neexistuje v media 
     */
    protected $missingFileFilePath = NULL;
    
    /**
     * @var array nastaveni db
     */
    protected $settings = NULL;

    /**
     * @var Logger 
     */
    private $log = NULL;
    
    
    function __construct(PDO $pdo, $dbType, $settings) {
        $this->pdo = $pdo;
        $this->dbType = $dbType;
        
        $this->settings = $settings;
        
        // nastavuje mista, kan se upladaji soubory spravovane db
        $this->accessabeFileDir = $settings['accessabeFileDir'];
        $this->hiddenFilesDir = $settings['hiddenFilesDir'];
        
        $this->daoImplCacheDir = $settings['daoImplCacheDir'];
        
        // defaultni hodnota 
        // TODO: asi by se melo predavat z konfigu
        $this->missingFileFilePath = __BASE_DIR__  ."/www/images/noimage.jpg";
        
        $this->namingStrategy = new DefaultNamingStrategy();
        
        $this->log = Logger::getLogger(self::class);
        
        // test jestli existuji potrebne soubory
        if( !file_exists($this->accessabeFileDir) ) {
            throw new Exception("DB managed visible files dir ". $this->accessabeFileDir ." doesn't exist.");
        }
        if( !file_exists($this->hiddenFilesDir) ) {
            throw new Exception("DB managed hidden files dir ". $this->hiddenFilesDir ." doesn't exist.");
        }
        
        // register close connection function on exiting script
        $_this = $this;
        register_shutdown_function(function() use ($_this) { $_this->close(); } );
    }
    
    /**
     * TODO: melo by se presunout do konstruktoru
     * @return type Description defaultni nastaveni pro vsechny db 
     */
    public function getDefaultSettings() {
        $defaultSetings = [
            "codepage" => 'utf8',
            "strictLoad" => false, // urcuje, jestli se ma striktne kontrolovat, ze se pri load neco nacetlo
            'daoImplCacheDir' => NULL,
            
        ];
        return $defaultSetings;
    }

    /**
     * TODO docasne, melo by se odstranit a odstinit zbytek aplikace od PDO
     * @return PDO
     */
    public function getPDO() {
        return $this->pdo;
    }

    /**
     * vraci jaka db je pripojena
     * @return string
     */
    public function getDbType() {
        return $this->dbType;
    }

    /**
     * strategie pro prevod nazvu objektu na tabulky
     * @return NamingStrategy
     */
    public function getNamingStrategy() {
        return $this->namingStrategy;
    }

    /**
     * @param NamingStrategy $namingStrategy
     */
    public function setNamingStrategy(NamingStrategy $namingStrategy) {
        $this->namingStrategy = $namingStrategy;
    }

    /**
     *
     * @see \IZON\DB\Dao::load()
     */
    public function load($className, $id) {
        $classDescription = $this->getClassDesctiption($className);
        $cache = $this->getSqlStatemetCache($className);
        
        if( !$cache->hasStatement('load') ) {
            $properties = '';
            foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
                if($properties != '') {
                    $properties .= ', ';
                }
                $properties .= $value .' as '. $key;
            }
            $sql = 'select '. $properties .' from '. $classDescription->getTableName() .' where '. $classDescription->getIdColumnName() .' = :id';
            
//            echo $sql ."\n";
            
            $pdo = $this->getPDO();
            $cache->setStatement('load', $pdo->prepare($sql));
       }
       
       $statement = $cache->getStatement('load');
       $statement->bindValue(":id", $id);
       $statement->execute();

       $retObj = $this->fetchObject($className, $statement);
       $statement->closeCursor();
       
       // test jestli se neco nacetlo
       if( $this->settings['strictLoad']
           && $retObj === FALSE ) {
           throw new Exception("Pro tridu $className a id $id v db neexistuje objekt.");
       }
       
       return $retObj;
    }

    /**
     *
     * @see \IZON\DB\GenericDao::loadAll()
     */
    public function loadAll($className) {

        $classDescription = $this->getClassDesctiption($className);
        $cache = $this->getSqlStatemetCache($className);
        
        if( !$cache->hasStatement('loadAll') ) {
           $properties = '';
           foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
               if($properties != '') {
                   $properties .= ', ';
               }
               $properties .= $value .' as '. $key;
           }
           $sql = 'select '. $properties .' from '. $classDescription->getTableName() .' limit :maxResults offset :firstResult';

           $pdo = $this->getPDO();

           $cache->setStatement('loadAll', $pdo->prepare($sql));
       }
       
       $statement = $cache->getStatement('loadAll');
       
       $queryResult = new QueryResult($this, $statement, $className, []);
       return $queryResult;
    }

    /**
     * 
     * @param class $className
     * @param array $params Parametry
     * @return QueryResult returns all object of class $className in undetermined order
     */
    public function find($className, array $params, array $orders = []) {
        if( !is_array($params) ) {
            throw new Exception("\$params for search must not be array");
        }
        
        $classDescription = $this->getClassDesctiption($className);
        $cache = $this->getSqlStatemetCache($className);
        
        // vytvori nazev pod jakym cachovat
        $findQueryName = 'find';
        $parameters = [];
        foreach($params as  $kay => $value) {
            $findQueryName .= "|". $kay;
            $parameters[":arg". $kay] = $value;
        }
        foreach($orders as $order => $asc) {
            $findQueryName .= "#". $order . ($asc ? "ASC" : "DESC");
        }
        
        if( !$cache->hasStatement($findQueryName) ) {
            $properties = '';
            foreach ($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
                if ($properties != '') {
                    $properties .= ', ';
                }
                $properties .= $value . ' as ' . $key;
            }
            $sql = 'select ' . $properties . ' from ' . $classDescription->getTableName();

            if( count($params) > 0 ) {
               $sql .= " where ";
            }
            $index = 0;
            // pridava nazvy parametru
            foreach ($params as $kay => $value) {
                if( $index++ != 0 ) {
                    $sql .= " and ";
                }
                $sql .= $classDescription->getColumnName($kay) ." = :arg". $kay ." ";
            }
            
            if( !\IZON\Arrays\isEmpty($orders) ) {
                $sql .= " order by ";
                $first = true;
                foreach($orders as $order => $asc) {
                    if( !$first ) {
                        $sql .= ", ";
                    } else {
                        $first = false;
                    }
                    $sql .=  $classDescription->getColumnName($order) ." ". ($asc ? "asc" : "desc");
                }
            }
            
            $sql .= ' limit :maxResults offset :firstResult';

            $pdo = $this->getPDO();

            $cache->setStatement($findQueryName, $pdo->prepare($sql));
       }
       
       $statement = $cache->getStatement($findQueryName);
       
       $queryResult = new QueryResult($this, $statement, $className, $parameters);
       return $queryResult;
    }
    
    /**
     *
     * @see \IZON\DB\GenericDao::update()
     */
    public function update($obj) {
        $className = get_class($obj);
        $classDescription = $this->getClassDesctiption($className);
        $cache = $this->getSqlStatemetCache($className);
        
        if( !$cache->hasStatement('update') ) {
            $properties = '';
            foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
                if( $key != $classDescription->getIdPropertyName()) { // updatuj vsechno krome id
                    if($properties != '') {
                        $properties .= ', ';
                    }
                    $properties .= $value ." = :". $key;
                }
            }
            
            if( $properties == '' ) { // no other properties than id, no need to update
                return;
            }
            
            $sql = 'update '. $classDescription->getTableName() .' set '. $properties .' where '. $classDescription->getIdColumnName() .' = :'. $classDescription->getIdPropertyName();
            $pdo = $this->getPDO();
            $cache->setStatement('update', $pdo->prepare($sql));
        }
        
        $statement = $cache->getStatement('update');
        
        foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            $this->bindSerializedValue($classDescription, $statement, $obj, $key);
        }
        
        $statement->execute();
        $statement->closeCursor();
        
        // updatuje soubory na disku
        // TODO: upravit, aby porvadelo i transakce
        // uklada soubory na disk
        // TODO: presunout to spravy transakci
        foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            if( $classDescription->isPropertyFile($key) ) { // updatuj soubory pokud nejake existuji
                
                $fileDescription = $classDescription->getPropertyDescription($key);
                /* @var $file File */
                $updateFile = $classDescription->getPropertyValue($obj, $key);
                $storageDir = $fileDescription->getStorageDir($obj);
                
                if( $updateFile != NULL
                    && !$this->isInFileStorage($updateFile) ) { // neni ve file storage, je ho tam potreba presunout
                    if( !file_exists($storageDir) ) { // vytvorit adresar pokud neexistuje
                        mkdir($storageDir, 0777, true); // vytvari se rekurzivne TODO: asi resit prava nejak lepe
                    }
                    // vymaz puvodni obsah adresare s obrazky
                    foreach(scandir($storageDir) as $file) {
                        if ('.' === $file || '..' === $file) continue;
                        else {
                            chmod("$storageDir/$file", 0777);
                            unlink("$storageDir/$file");
                        }
                    }
                    $safeFileName = File::getSafeFileName($updateFile->getFileName());
                    \IZON\File\copy($updateFile, $storageDir ."/". $safeFileName); // zkopirovat do file storage viditelnefo z web
//                    $updateFile->delete();
                    
                    // nacti do objektu ulozeny soubor
                    $movedFile = $fileDescription->parse($safeFileName, $obj);
                    $classDescription->setPropertyValue($obj, $key, $movedFile);
                }
            }
        }
    }

    /**
     * @see DBConnection::save()
     */
    public function save($obj) {
        $pdo = $this->getPDO();
        
        $className = get_class($obj);

        /* @var $variable ClassName */
        $classDescription = $this->getClassDesctiption($className);
        $cache = $this->getSqlStatemetCache($className);

        if( !$cache->hasStatement('insert') ) {
            $sql = $this->createInsertSQLString($classDescription);

            $cache->setStatement('insert', $pdo->prepare($sql));
        }
        
        $statement = $cache->getStatement('insert');
        
        foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            if( $key != $classDescription->getIdPropertyName()) { // id neni nastavene
                $this->bindSerializedValue($classDescription, $statement, $obj, $key);
            }
        }
        // vykona pripraveny statement
        $statement->execute();
        
        // ziska posledni id vlozeneho zaznamu
        $idPropertyName = $classDescription->getIdPropertyName();
        $idValue = $this->getLastInsertedId($pdo, $statement);
        $classDescription->setPropertyValue($obj, $idPropertyName, $idValue); // ulozi ho do objektu
        
        // uzavrit cursor pro dotaz
        $statement->closeCursor();
        
        // uklada soubory na disk
        // TODO: napred vymazat adresar ve kterem se soubor nachazi
        // TODO: presunout to spravy transakci
        foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            if( $classDescription->isPropertyFile($key) ) { // mazej soubory pokud nektere existuji
                $fileDescription = $classDescription->getPropertyDescription($key);
                $storageDir = $fileDescription->getStorageDir($obj);
                /* @var $file File */
                $file = $classDescription->getPropertyValue($obj, $key);
                if( $file != NULL ) {
                    if( !file_exists($storageDir) ) { // vytvorit adresar pokud neexistuje
                        mkdir($storageDir, 0777, true); // vytvari se rekurzivne
                    }
                    $safeFileName = File::getSafeFileName($file->getFileName());
                    \IZON\File\copy($file, $storageDir ."/". $safeFileName); // zkopirovat do file storage viditelnefo z web
//                    $file->delete();
                    
                    $movedFile = $fileDescription->parse($safeFileName, $obj);
                    $classDescription->setPropertyValue($obj, $key, $movedFile);
                }
            }
        }

        return $idValue;
    }
    
    /**
     * vytvori string pro insert do tabulty pro tridu definovanou pomoci $classDescription
     * @param ClassDescription $classDescription trida pro kterou se ma vytvorit sql
     * @return string
     */
    protected abstract function createInsertSQLString(ClassDescription $classDescription);
    
    /**
     * vraci vygenerovane id posledniho vlozene zaznamu
     * @param PDO $pdo pdo db pripojeni
     * @param PDOStatement $statement statement ktery byl vykonan 
     * @return id
     */
    protected abstract function getLastInsertedId(PDO $pdo, PDOStatement $statement);

    /**
     *
     * @see \IZON\DB\GenericDao::delete()
     */
    public function delete($obj, $className = NULL) {
        if( is_object($obj) ) { // je objekt, ziskat tridu pro smazani
            $className = get_class($obj);
        } else { // neni objekt to znamena ze je id, musi byt nastaveno $className
            if( $className == NULL ) {
                throw new \Exception("Not valid $className");
            }
        }
        /* @var $classDescription ClassDescription */
        $classDescription = $this->getClassDesctiption($className);
        $cache = $this->getSqlStatemetCache($className);
        
        if( !$cache->hasStatement('delete') ) {
            $sql = 'delete from '. $classDescription->getTableName() .' where '. $classDescription->getIdColumnName() .' = :id';
            $pdo = $this->getPDO();
            $cache->setStatement('delete', $pdo->prepare($sql));
        }
        
//        echo $sql;
        $statement = $cache->getStatement('delete');
        
        $id = $obj;
        if( is_object($obj) ) {
            $idPropName = $classDescription->getIdPropertyName();
            $id = $classDescription->getPropertyValue($obj, $idPropName);
        }
        
        $statement->bindValue(":id", $id);
        $statement->execute();
        $statement->closeCursor();
        
        // maze soubory na disku
        // TODO: presunout do spravy transakci
        // FIXME: pokud je id smazat soubory, ted se spravne provede jen pro objekt
        if( is_object($obj) ) {
            $objectStorageDirs = [];
            
            foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
                if( $classDescription->isPropertyFile($key) ) { // mazej soubory pokud nektere existuji
                    $fileDescription = $classDescription->getPropertyDescription($key);
                    $storageDir = $fileDescription->getStorageDir($obj);
                    $objectStorageDirs[] = $fileDescription->getObjectStorageDir($obj);
                            
                    // smaze soubor
                    $file = $classDescription->getPropertyValue($obj, $key);
                    if( $file != NULL ) { // property je NULL tj. neexistuje soubor na disku
                        $file->delete();
                    }
                    
                    if( file_exists($storageDir) ) {
                        // vymaze obsah adresare
                        foreach(scandir($storageDir) as $file) {
                            if ('.' === $file || '..' === $file) continue;
                            else {
                                chmod("$storageDir/$file", 0777);
                                unlink("$storageDir/$file");
                            }
                        }
                        // smazat adresar
                        rmdir($storageDir);
                    }
                }
            }
            
            // smazat adresar pro dany objekt 
            foreach($objectStorageDirs as $dir) {
                if(file_exists($dir) ) {
                    rmdir($dir);
                }
            }
        }
    }
    
    public function close() {
        $this->sqlStatementsCache = [];
        $this->objectCache = [];
        $this->classDescriptions = [];
        
        // TODO: co jeste uvolnit
        $this->pdo = NULL;
    }
    
    /**
     * zapocne transakci
     */
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }
    
    /**
     * commituje transakci
     */
    public function commit() {
        return $this->pdo->commit();
    }
    
    /**
     * rollbackuje transakci
     */
    public function rollBack() {
        return $this->pdo->rollBack();
    }


    /**
     * 
     * @param string|class $className
     * @param QueryFactory|array $qureyFactory
     * @param string $methodName
     * @param array $parameters
     * @return QueryResult
     */
    public function executeFind($className, $qureyFactory, $methodName, $parameters) {

        $paramsCount = count($parameters);

        $classDescription = $this->getClassDesctiption($className);
        $queryHelpers = new QueryHelpers($classDescription, $this, $parameters);
        $queryString = $classDescription->getQueryString($qureyFactory, $methodName, $paramsCount, $queryHelpers);
        
        if( strstr($queryString, ':maxResults') === false ) {
            throw new Exception("Dotaz musi mit limit :maxResults");
        }
        if( strstr($queryString, ':firstResult') === false ) {
            throw new Exception("Dotaz musi mit offset :firstResult");
        }
        
        
        $pdo = $this->getPDO();
        $statement = $pdo->prepare($queryString);
        $queryResult = new QueryResult($this, $statement, $className, $queryHelpers->getQueryParameters());

        return $queryResult;
    }
    
    /**
     * provede count
     * @param class $className
     * @param string $methodName
     * @param array $parameters
     */
    public function executeCount($className, $qureyFactory, $methodName, $parameters) {
        
        $paramsCount = count($parameters);
        
        $classDescription = $this->getClassDesctiption($className);
        $queryHelpers = new QueryHelpers($classDescription, $this, $parameters);
        $queryString = $classDescription->getQueryString($qureyFactory, $methodName, $paramsCount, $queryHelpers );
        
        $pdo = $this->getPDO();
        $statement = $pdo->prepare($queryString);
        
        foreach($queryHelpers->getQueryParameters() as $key => $value) {
            $statement->bindValue($key, $value);
        }
        
        $statement->execute();
//        var_dump($statement);
        $countArray = $statement->fetch(PDO::FETCH_ASSOC);
//        var_dump($countArray);
        if( !isset($countArray[$queryHelpers->getCountAlias()]) ) {
            throw new Exception("count dotaz musi mit pocet radku pod aliasem ". $queryHelpers->getCountAlias());
        }
        
        $statement->closeCursor();
        
        return $countArray[$queryHelpers->getCountAlias()];
    }

    /**
     * provede find dotaz
     * @param class $className
     * @param string $methodName
     * @param array $parameters
     */
    public function executeCustomFind($className, $qureyFactory, $methodName, array $parameters, array $queryTypeParameters) {
        
        $classDescription = $this->getClassDesctiption($className);
        $queryHelpers = new QueryHelpers($classDescription, $this, $parameters);
        $queryString = $classDescription->getCustomQueryString($qureyFactory, $methodName, $parameters, $queryTypeParameters, $queryHelpers);
        $pdo = $this->getPDO();
        $queryResult = new QueryResult($this, $pdo->prepare($queryString), $className, $queryHelpers->getQueryParameters());
        
        return $queryResult;
    }
    
    /**
     * vyt
     * @param string $className jmeno tridy nad kterou se provadi dotaz
     * @param type $qureyFactory
     * @param type $methodName
     * @param array $parameters
     * @param array $queryTypeParameters puvodne bylo na slozeni order by, ale spis budou obecne jakekoli parametry potrebne pro slozeni dotazu
     * @return type
     * @throws Exception
     */
    public function executeCustomCount($className, $qureyFactory, $methodName, array $parameters, array $queryTypeParameters) {
        
        $classDescription = $this->getClassDesctiption($className);
        $queryHelpers = new QueryHelpers($classDescription, $this, $parameters);
        $queryString = $classDescription->getCustomQueryString($qureyFactory, $methodName, $parameters, $queryTypeParameters, $queryHelpers);
        
        $pdo = $this->getPDO();
        $statement = $pdo->prepare($queryString);
        
        foreach($queryHelpers->getQueryParameters() as $key => $value) {
            $statement->bindValue($key, $value);
        }
        
        $statement->execute(); 
        $countArray = $statement->fetch(PDO::FETCH_ASSOC);
        if( !isset($countArray[$queryHelpers->getCountAlias()]) ) {
            throw new Exception("count dotaz musi mit pocet radku pod aliasem ". $queryHelpers->getCountAlias());
        }
        
        $statement->closeCursor();
        
        return $countArray[$queryHelpers->getCountAlias()];
    }
    
    /**
     * provede delete dotaz
     * @param type $className
     * @param type $methodName
     * @param type $parameters
     */
    public function executeDelete($className, $qureyFactory, $methodName, $parameters) {
        
        $paramsCount = count($parameters);
        $cache = $this->getSqlStatemetCache($className);
        $statementCashName = $methodName ."_". $paramsCount;
        
        if( !$cache->hasStatement($statementCashName) ) {
            $classDescription = $this->getClassDesctiption($className);
            $queryHelpers = new QueryHelpers($classDescription, $this, $parameters);
            $queryString = $classDescription->getQueryString($qureyFactory, $methodName, $paramsCount, $queryHelpers);
            
            $pdo = $this->getPDO();
            $cache->setStatement($statementCashName, $pdo->prepare($queryString));
        }
       
        $statement = $cache->getStatement($statementCashName);
        
        foreach($queryHelpers->getQueryParameters() as $key => $value) {
            $statement->bindValue($key, $value);
        }
        
        $statement->execute(); 
        $affectedRows = $statement->rowCount();
        $statement->closeCursor();
        
        return $affectedRows;
    }
    
    /**
     * provede update dotaz
     * @param type $className
     * @param type $methodName
     * @param type $parameters
     */
    public function executeUpdate($className, $queryFactory, $methodName, $parameters) {
        
        $paramsCount = count($parameters);
        $cache = $this->getSqlStatemetCache($className);
        $statementCashName = $methodName ."_". $paramsCount;
        
        if( !$cache->hasStatement($statementCashName) ) {
            $classDescription = $this->getClassDesctiption($className);
            $queryHelpers = new QueryHelpers($classDescription, $this, $parameters);
            $queryString = $classDescription->getQueryString($queryFactory, $methodName, $paramsCount, $queryHelpers);
            
            $pdo = $this->getPDO();
            $cache->setStatement($statementCashName, $pdo->prepare($queryString));
        }
       
        $statement = $cache->getStatement($statementCashName);
        
        foreach($queryHelpers->getQueryParameters() as $key => $value) {
            $statement->bindValue($key, $value);
        }
        
        $statement->execute(); 
        $affectedRows = $statement->rowCount();
        $statement->closeCursor();
        
        return $affectedRows;
    }
    
    public function paginate($className, PaginatorFactory $paginatorFactory, PaginatorConfig $config) {
        $pdo = $this->getPDO();
        $classDescription = $this->getClassDesctiption($className);
        
        // vytvari count dota
        $queryHelpers =  $queryHelpers = new QueryHelpers($classDescription, $this, $config->getParameters());
        $countQueryString = $paginatorFactory->count($queryHelpers, $config);
        
        // todo cachovat statement
        $countStatement = $pdo->prepare($countQueryString);
        foreach($queryHelpers->getQueryParameters() as $key => $value) {
            $countStatement->bindValue($key, $value);
        }
        $countStatement->execute();
        if ($row = $countStatement->fetch(PDO::FETCH_ASSOC)) {
            $resultsCount = $row[$queryHelpers->getCountAlias()];
        } else {
            throw new Exception("Count query error");
        }
        $countStatement->closeCursor();

        // vytvari dotaz na nacteni stranky
        $queryHelpers =  $queryHelpers = new QueryHelpers($classDescription, $this, $config->getParameters());
        $pageQueryString = $paginatorFactory->page($queryHelpers, $config);
        
        // TODO: cachovat statement
        $pageStatement = $pdo->prepare($pageQueryString);
        foreach($queryHelpers->getQueryParameters() as $key => $value) {
            $pageStatement->bindValue($key, $value);
        }
//        var_dump($config->getFirstResult());
        $pageStatement->bindValue(":maxResults", $config->getMaxPageSize(), PDO::PARAM_INT);
        $pageStatement->bindValue(":firstResult", $config->getFirstResult(), PDO::PARAM_INT);

        $pageStatement->execute();
        $page = $this->fetchObjectArray($className, $pageStatement);
        $pageStatement->closeCursor();
        
//        var_dump($page);
        return new PageContent($resultsCount, $config->getMaxPageSize(), $page);
    }


    /**
     * fetches one object from PDO statement result
     * @param class $className
     * @param PDOStatement $statement
     */
    public function fetchObject($className, $statement) {
        $row = $statement->fetch(PDO::FETCH_ASSOC);
        if( $row === FALSE)
            return FALSE;
        
        $classDescription = $this->getClassDesctiption($className);
        
        $retObj = new $className();
        
        // nastavit id jako prvni v tride
        $idPropertyName = $classDescription->getIdPropertyName();
        $idPropertyDescription = $classDescription->getPropertyDescription($idPropertyName);
        try {
            $propertyValue = $idPropertyDescription->parse( $row[$idPropertyDescription->getColumnReturnName()], $retObj );
        } catch (Exception $ex) {
            throw new Exception("Trida ". $classDescription->getClassName() ." podle definice property: ". $ex->getMessage());
        }
        $classDescription->setPropertyValue($retObj, $idPropertyName, $propertyValue);
        
        foreach( $classDescription->getPropertiesToColumnsTranslation() as $key => $value ) {
            if( $key != $idPropertyName ) { // id jiz neni protreba novu parsovat
                $propertyDescription = $classDescription->getPropertyDescription($key);

                try {
                    $propertyValue = $propertyDescription->parse( $row[$propertyDescription->getColumnReturnName()], $retObj );
                } catch (Exception $ex) {
                    throw new Exception("Trida ". $classDescription->getClassName() ." podle definice property: ". $ex->getMessage());
                }
                $classDescription->setPropertyValue($retObj, $key, $propertyValue);
            }
        }
        
        return $retObj;
    }
    
    /**
     * vrati pole objektu
     * @param type $className
     * @param type $statement
     * @return type
     */
    public function fetchObjectArray($className, $statement) {
        $res = [];
        while($obj = $this->fetchObject($className, $statement) ) {
//            var_dump($obj);
            $res[] = $obj;
        }
        return $res;
    }


    /**
     * vrati popis tridy, ulozi ho pokud neexistuje
     * @param class $className
     * @return ClassDescription Description
     */
    public function getClassDesctiption($className) {
        // neexistuje, vytvorit
        if( !isset($this->classDescriptions[$className]) ) {
            $this->classDescriptions[$className] = new ClassDescription($this, $className);
        }
        return $this->classDescriptions[$className];
    }
    
    /**
     * @var ClassDescription $classDescription description of class property belongs to
     * @var array $propertyDefinitions definitions 
     */
    public abstract function getPropertyDescriptions(ClassDescription $classDescription, $propertyDefinitions, $propertyTypes);
    
    /**
     * do statementu nastavi hodnotu parametru
     * @param type $classDescription
     * @param type $statement
     * @param type $obj
     * @param type $propertyName
     */
    protected function bindSerializedValue($classDescription, 
                                                $statement,
                                                $obj, 
                                                $propertyName) {
        $propertyValue = $classDescription->getPropertyValue($obj, $propertyName);
        $propertyDescription = $classDescription->getPropertyDescription($propertyName);
        
        try {
            $serializedPropertyValue = $propertyDescription->serialize($propertyValue, $obj);
        } catch (Exception $ex) {
            throw new Exception("Trida ". $classDescription->getClassName() ." property: ". $ex->getMessage());
        }
        
        $statement->bindValue(":$propertyName", 
                                $serializedPropertyValue, 
                                $propertyDescription->getPdoDataType($propertyValue));
    }


    /**
     * 
     * @param class $className
     * @return SqlStatementCache Description
     */
    protected function getSqlStatemetCache($className) {
        // neexistuje, vytvorit
        if( !isset($this->sqlStatementsCache[$className]) ) {
            $this->sqlStatementsCache[$className] = new SqlStatementCache();
        }
        return $this->sqlStatementsCache[$className];
    }
    
    /**
     * vraci ture pokud je soubor v db storage
     * @param File $file
     * @return type
     */
    protected function isInFileStorage(File $file) {
        return \IZON\File\isInSubDir($file->getFsPath(), $this->accessabeFileDir)
                || \IZON\File\isInSubDir($file->getFsPath(), $this->hiddenFilesDir);
//        return \IZON\String\startsWith(  str_replace("\\", "/", $file->getFsPath()), str_replace("\\", "/", $this->accessabeFileDir) )
//                || \IZON\String\startsWith( str_replace("\\", "/", $file->getFsPath()), str_replace("\\", "/", $this->hiddenFilesDir) );
    }
    
    public function __toString () {
        return self::class .'';
    }
    

/// vygenerovane gettery a settery
    function getDaoImplCacheDir() {
        return $this->daoImplCacheDir;
    }

    function setDaoImplCacheDir($daoImplCacheDir) {
        $this->daoImplCacheDir = $daoImplCacheDir;
    }
}