<?php

namespace IZON\DB\Utils\PropertyDescriptions;

use \PDO;
use \Exception;

use IZON\Logs\Logger;

use IZON\IO\File;

use IZON\DB\Utils\ClassDescription;

/**
 * Popis pukladani soubou
 */
class FileDescription extends AbstractPropertyDescription {

    /**
     * ma byt primo viditelny na webu
     * @var boolean 
     */
    protected $webVisible = true;
    
    /**
     * v jakem podadresari maji byt ulozeny soubory pro tridu
     * @var string 
     */
    protected $fileDir = NULL;

    /**
     * v jakem adresari se nachazeji soubory pro dane property
     * @var string
     */
    protected $storageDir = NULL;
    
    /**
     * @var string jak se ma jmenovat obrazek, co se nahradi pokud obrazk neexistuje v media 
     */
    protected $missingFileFilePath = NULL;
    
    /**
     *
     * @var Logger 
     */
    protected $log;

    /**
     * 
     * @param string $propertyName
     * @param string $columnName
     * @param boolean $notNull
     * @param array $parameters
     */
    function __construct(ClassDescription $classDesctiption,
                            $propertyName, $columnName, $columnReturnName, $notNull, array $parameters = []) {
        $this->webVisible = $this->extractParameter("webVisible", $parameters, false, true);
        // 
        $defaultFileDir = str_replace("_", "-", $classDesctiption->getTableName());
        $this->fileDir = $this->extractParameter("fileDir", $parameters, false, $defaultFileDir);
        
        if($this->webVisible) {
            $this->storageDir = $this->extractParameter("accessabeFileDir", $parameters, true);
        } else {
            $this->storageDir = $this->extractParameter("hiddenFilesDir", $parameters, true);
        }
        
        $this->missingFileFilePath = $this->extractParameter("missingFileFilePath", $parameters, true);
        
        $this->storageDir .= "/". $this->fileDir;
        $this->storageDir = \IZON\File\normalizeFileName($this->storageDir);
        
        parent::__construct($classDesctiption, $propertyName, $columnName, $columnReturnName, $notNull, $parameters);

        $this->dbDataType = File::class;
        $this->pdoDataType = PDO::PARAM_STR;
        $this->ansiDataType = AbstractPropertyDescription::ANSI_DATA_TYPE_BIT_VARYING;
        
        $this->log = Logger::getLogger(__CLASS__);
    }

    public function parse($value, $object) {
        if ($value == NULL) {
            if ($this->getNotNull()) {
                throw new Exception("" . $this->getPropertyName() . " nesmi byt null");
            }
            return NULL;
        }
        
        $storageDir = $this->getStorageDir($object);
        $fileName = $storageDir . "/" . $value;
        if ( !file_exists($fileName) ) {
            $this->log->error("Db file " . $fileName . " isn't present on filesystem.");
            
            if( file_exists($this->missingFileFilePath) ) {
                $fileName = $this->missingFileFilePath;
            } else {
                $this->log->error("Rplacement form missing file " . $this->missingFileFilePath . " isn't present on filesystem.");
                return NULL;
            }
            
        }
        // create file
        $file = new File($fileName);

        return $file;
    }

    /**
     * 
     * @param File $value
     * @return string
     * @throws Exception
     */
    public function serialize($value, $object) {
        if ($value == NULL) {
            if ($this->getNotNull()) {
                throw new Exception("" . $this->getPropertyName() . " nesmi byt null");
            }
            return NULL;
        }
        if (!($value instanceof File)) {
            throw new Exception("Hodnota musi byt null nebo " . File::class);
        }
        
        $serializedString =  $value->getFileName();
        if( !$this->isInFileStorage($value, $object) ) {
            $serializedString = File::getSafeFileName($serializedString); // vem bezpecne jmeno
        }
        
        return $serializedString;
    }

    function getPdoDataType($value) {
        if ($value == NULL)
            return PDO::PARAM_NULL;
        return $this->pdoDataType;
    }

    function isWebVisible() {
        return $this->webVisible;
    }
    
    /**
     * vrati informace do ktereho adresare souboru se ma nahrat
     * @param object $object
     */
    public function getStorageDir($object) {
        $propertyDir = str_replace("_", "-", $this->columnName);
        return $this->getObjectStorageDir($object) ."/". $propertyDir;
    }
    
    /**
     * vrati informace do ktereho adresare se maji nahrat soubory pro objekt
     * @param object $object
     */
    public function getObjectStorageDir($object) {
        $idPropertyName = $this->classDesctiption->getIdPropertyName();
        $objectId = $this->classDesctiption->getPropertyValue($object, $idPropertyName);
        
        return $this->storageDir ."/". $objectId;
    }
    
    /**
     * vraci ture pokud je soubor v uz soubor ulozeni ve storage
     * @param File $file
     * @return type
     */
    protected function isInFileStorage(File $file, $object) {
        $getObjectStorageDir = $this->getObjectStorageDir($object);
                
        return \IZON\String\startsWith( str_replace("\\", "/", $file->getFsPath()), str_replace("\\", "/", $getObjectStorageDir) );
    }
}