<?php

namespace IZON\DB\Utils;

use \Exception;
use \ReflectionClass;

use \PhpDocReader\PhpDocReader;

use IZON\Logs\Logger;

use \IZON\DB\Impl\DBConnectionCommon;
use \IZON\DB\NamingStrategy;
use \IZON\DB\DBObject;
use \IZON\DB\QueryFactory;
use \IZON\DB\Utils\PropertyDescriptions\AbstractPropertyDescription;

use IZON\DB\Utils\PropertyDescriptions\FileDescription;
use IZON\DB\Utils\PropertyDescriptions\ImageDescription;

/**
 * Obsahuje popis tridy potrebny pro vytvoreni dotazu
 * a spetne naplneni objektu a podobjedku a kolekci
 */
class ClassDescription {
    
    /**
     * pripojeni k db
     * @var DBConnectionCommon 
     */
    protected $dbConnection;


    /**
     * jak se maji prekladat nazvy
     * @var NamingStrategy 
     */
    protected $namingStrategy;
    
    /**
     * nazev tridy, kterou tento objekt popisuje
     * @var string 
     */
    protected $className;
    
    /**
     * @var ReflectionClass
     */
    protected $classReflectionClass;

    /**
     * @var array ReflectionProperty hash obsahujici ReflectionProperty k jmenu property
     */
    protected $reflectionProperties;

    /**
     * jmeno tabulky na kterou se ma mapovat
     * @var string 
     */
    protected $tableName;
    
    /**
     * descriptions of properties of object
     * subclasses of AbstractPropertyDescription 
     * @var array 
     */
    protected $propertyDescriptions = [];

    /**
     * preklad jmen properties na odpovidajici jmena sloupcu
     * @var array 
     */
    protected $propertiesToColumnsTranslation = [];
    
    /**
     * jak se jmenuje property identifikujici objekt, defaultne je to id
     * @var string 
     */
    protected $idPropertyName = "id";
    
    /**
     * objekt slouzici k vytvareni dotazu
     * @var QueryFactory 
     */
//    protected $queryFactory = NULL;
    
    /**
     * 
     * @var Logger 
     */
    protected $log = NULL;
    
    /**
     * nacita datovy typ z dokumentacniho bloku
     * @var PhpDocReader 
     */
    protected static $docReader = NULL;

    /**
     * skalarni typy promenych v php
     * http://php.net/manual/en/language.types.intro.php
     * @var array 
     */
    protected static $scalarTypes = array(
        'bool',
        'boolean',
        'string',
        'int',
        'integer',
        'float',
        'double',
        'number' //  integer or float
    );

    /**
     * 
     * @param DBConnectionCommon $dbConnection
     * @param DBObject $className
     * @throws Exception
     */
    public function __construct(DBConnectionCommon $dbConnection, 
                                $className) {
        // sprovoznit logovani
        $this->log = Logger::getLogger(self::class);

        if( !class_exists($className) ) { // tests if class is DBObject
            throw new Exception("Class $className does exist.");
        }
        
        if( !is_subclass_of($className, DBObject::class) ) { // tests if class is DBObject
            throw new Exception("$className does not implement ". DBObject::class);
        }
        
        $this->dbConnection = $dbConnection;

        $this->namingStrategy = $dbConnection->getNamingStrategy();
        
        $this->classReflectionClass = new ReflectionClass($className);
        $this->className = $className;
        
        // nastaveni jmena tabulky
        if( method_exists($this->className, "__tableName") ) {
            $this->tableName = $className::__tableName();
        } else {
            $this->tableName = $this->namingStrategy->classToTableName($this->className);
        }
        // nastaveni prekladu jmena property na jmeno sloupce
        $this->initProperities();
        
//        // nacte potrebnou query factory
//        $queryFactoryName = str_replace("\\Domain\\", "\\DB\\", $className) ."QueryFactory". $this->dbConnection->getDbType();
//        if( class_exists($queryFactoryName) ) {
//            $this->queryFactory = new $queryFactoryName();
//        } else {
//            $queryFactoryName = str_replace("\\Domain\\", "\\DB\\", $className) ."QueryFactory";
//            if( class_exists($queryFactoryName) ) {
//                $this->queryFactory = new $queryFactoryName();
//            }
//        }
        // TODO: test jestli query factory vyhovuje parametrum
    }
    
    function getClassName() {
        return $this->className;
    }
    
    /**
     * vrati nazev tabulky pro dany objekt
     * @return type
     */
    public function getTableName() {
        return $this->tableName;
    }

    /**
     * vrati preklad vsech poperit objektu na jmena sloupcu
     * @return array
     */
    public function getPropertiesToColumnsTranslation() {
        return $this->propertiesToColumnsTranslation;
    }

    /**
     * nazev property, ktera obsahuje id objektu
     * @return type
     */
    public function getIdPropertyName() {
        return $this->idPropertyName;
    }
    
    /**
     * nazev sloupce, ktery obsahuje id
     * @return type
     */
    public function getIdColumnName() {
        return $this->propertiesToColumnsTranslation[$this->idPropertyName];
    }
    
    public function getColumnName($propertyName) {
        if( !isset($this->propertiesToColumnsTranslation[$propertyName]) ) {
            throw new \Exception("Trida ". $this->className . " nema property ". $propertyName);
        }
        return $this->propertiesToColumnsTranslation[$propertyName];
    }
    
    /**
     * returns value of property with name $propertyName from $object
     * @param object $object
     * @param string $propertyName
     * @return mixed Description
     */
    public function getPropertyValue($object, $propertyName) {
        if( !isset($this->reflectionProperties[$propertyName]) ) {
            throw new Exception("Trida ". $this->className ." nema property ". $propertyName);
        }
        
        $reflectionProperty = $this->reflectionProperties[$propertyName];
//        $reflectionProperty->setAccessible(true);
        return $reflectionProperty->getValue($object);
    }
    
    /**
     * 
     * @param mixed $object object property to be set to
     * @param strinc $propertyName name of the property to be set
     * @param mixed $propertyValue value of property to be set
     */
    public function setPropertyValue($object, $propertyName, $propertyValue) {
        if( !isset($this->reflectionProperties[$propertyName]) ) {
            throw new Exception("Trida ". $this->className ." nema property ". $propertyName);
        }
        
        $reflectionProperty = $this->reflectionProperties[$propertyName];
        $reflectionProperty->setValue($object, $propertyValue);
    }

    /**
     * pro danou metodu vrati retezec sql dotazu
     * @param type $methodName
     * @param type $parametersCount
     * @param type $queryHelpers
     * @return type
     * @throws \Exception
     */
    public function getQueryString($qureyFactory, $methodName, $parametersCount, $queryHelpers) {
//        var_dump($qureyFactory);
        $factoryMethodName = $methodName ."_". $parametersCount;
        if( !is_array($qureyFactory) ) {
//            var_dump($qureyFactory);
            if( $qureyFactory === NULL ) {
                throw new \Exception("Pro tridu ". $this->className ." neexistuje query factory");
            }
            if(!method_exists($qureyFactory, $factoryMethodName) ) {
                throw new \Exception("Pro tridu ". $this->className ." neexistuje metoda $factoryMethodName pro vytvoreni dotazu.");
            }

            $queryString = $qureyFactory->$factoryMethodName($queryHelpers);
        } else {
//            var_dump($qureyFactory[$factoryMethodName]);
            if( !isset($qureyFactory[$factoryMethodName]["queryFactory"]) ) {
                throw new \Exception("Pro tridu ". $this->className ." neexistuje metoda $factoryMethodName pro vytvoreni dotazu.");
            }
            $factoryCallable = $qureyFactory[$factoryMethodName]["queryFactory"];
            $queryString = $factoryCallable($queryHelpers);
        }
        return $queryString;
    }
    
    /**
     * pro danou metodu vrati retezec sql dotazu
     * @param type $methodName
     * @param type $parametersCount
     * @param type $queryHelpers
     * @return type
     * @throws \Exception
     */
    public function getCustomQueryString($qureyFactory, $methodName, array $parameters, array $queryTypeParameters, $queryHelpers) {
        $factoryMethodName = $methodName;
        if( !is_array($qureyFactory)
            && $qureyFactory instanceof QueryFactory) {
//            var_dump($qureyFactory);
            if( $qureyFactory === NULL ) {
                throw new Exception("Pro tridu ". $this->className ." neexistuje query factory");
            }
            if(!method_exists($qureyFactory, $factoryMethodName) ) {
                throw new Exception("Pro tridu ". $this->className ." neexistuje metoda $methodName pro vytvoreni dotazu.");
            }
            return $qureyFactory->$factoryMethodName($queryHelpers, $parameters, $queryTypeParameters);
        } else if(is_array($qureyFactory)) {
//            var_dump($qureyFactory[$factoryMethodName]);
            if( !isset($qureyFactory[$factoryMethodName]["queryFactory"]) ) {
                throw new Exception("Pro tridu ". $this->className ." neexistuje metoda $methodName pro vytvoreni dotazu.");
            }
            $factoryCallable = $qureyFactory[$factoryMethodName]["queryFactory"];
            return $factoryCallable($queryHelpers, $parameters, $queryTypeParameters);
        } else {
            throw new Exception("Neni ani QueryFactory ani pole s query factory");
        }
    }
    
    /**
     * 
     * @param type $propertyName
     * @return AbstractPropertyDescription
     * @throws Exception
     */
    public function getPropertyDescription($propertyName) {
        if( !isset($this->propertyDescriptions[$propertyName]) )
            throw new Exception("Popis $propertyName neni definovan.");
        return $this->propertyDescriptions[$propertyName];
    }
    
    /**
     * @param string $className Description
     * @param string $propertyName
     * @returns true if property is filek 
    */
    public function isPropertyFile($propertyName) {
        $propertyDescription = $this->getPropertyDescription($propertyName);
        return $propertyDescription instanceof FileDescription
                || $propertyDescription instanceof ImageDescription;
    }

    /**
     * zpracuje pojmenovani properit
     *
     * Methdo expected - Class with $className exist
     * @param class $className
     */
    protected function initProperities() {
        // go thought all properties
        $propertiesHash = [];
        foreach($this->classReflectionClass->getProperties() as $property) {
            $propertiesHash[$property->getName()] = $property;
        }
        //go thought all parents properties
        $reflectionClass = $this->classReflectionClass;
        while($reflectionClass->getParentClass()){
            $reflectionClass = $reflectionClass->getParentClass();
            foreach($reflectionClass->getProperties() as $property) {
                if( !isset($propertiesHash[$property->getName()]) ) {
                    $propertiesHash[$property->getName()] = $property;
                }
            }
        }
        $this->reflectionProperties = $propertiesHash;
        
        // definice jednotlivych 
        $propertyDefinitions = [];
        $propertyTypes = [];
        foreach($propertiesHash as $property) {
            // vsechny nastavit aby byly pristupne
            $property->setAccessible(true);
            
            // zjstit jentli ma definici k vride
            $propertyDefinition = NULL;
            if( $this->classReflectionClass->hasMethod("__". $property->getName()) ) {
                $method = $this->classReflectionClass->getMethod("__". $property->getName());
                $propertyDefinition = $method->invoke(NULL);
            }
            
            $propertyDefinitions[$property->getName()] = $propertyDefinition;
            $propertyTypes[$property->getName()] = $this->getCommentDataType($property);
        }
        
        $this->propertyDescriptions = $this->getPropertyDescriptions($propertyDefinitions, $propertyTypes);
        
        foreach($this->propertyDescriptions as $key => $propertyDescription) {
            // naplni preklady properit na soubory
            $this->propertiesToColumnsTranslation[$key] = $propertyDescription->getColumnName();
        }
    }
    
    /**
     * vytahne datovy typ z komentare u property v mapovane tride
     * @param type $property
     * @return type
     */
    protected function getCommentDataType($property) {
        $reader = self::getDocReader();
        //find existing class
        $dataType = $reader->getPropertyClass($property); // nacte tridu
        //existing class not found -> found scalar tipes
        if( $dataType == NULL ) {
            if (preg_match('/@var\s+([^\s]+)/', $property->getDocComment(), $matches)) { // nacte skalarni typ
                list(, $dataType) = $matches;
                // pouzivej jen skalarni typy
                if ( !in_array($dataType, self::$scalarTypes) ) {
                    $dataType = NULL;
                }
            } else {
                $dataType = NULL;
            }
        }

        return $dataType;
    }
    
    /**
     * 
     * @param AbstractDefinition $propertyDefinition
     * @param class|string $propertyType
     */
    protected function getPropertyDescriptions($propertyDefinitions, $propertyTypes) {
        return $this->dbConnection->getPropertyDescriptions($this, $propertyDefinitions, $propertyTypes);
    }

    protected function getDocReader() {
        //TODO: set new phpDocReader
        if( self::$docReader == NULL ) {
            self::$docReader = new PhpDocReader();
        }
        return self::$docReader;
    }


    
}
