<?php

namespace IZON\DB\Files\Proxies;

use Exception;
use IZON\IO\Exceptions\UnsupportedRasterImageTypeException;
use IZON\IO\FileInterface;
use IZON\IO\RasterImageInterface;
use League\Flysystem\FilesystemInterface;

/**
 * proxy object that represents
 */
class RasterImageProxy extends FileProxy implements RasterImageInterface
{
    /**
     * @var int
     */
    protected int $pixelWidth;

    /**
     * @var int
     */
    protected int $pixelHeight;


    public function __construct(
        string $fileName,
        int $fileSize,
        string $mimeType,
        int $timestamp,
        int $pixelWidth,
        int $pixelHeight
    ) {
        parent::__construct($fileName, $fileSize, $mimeType, $timestamp);
        $this->pixelWidth = $pixelWidth;
        $this->pixelHeight = $pixelHeight;
    }

    /**
     * @param RasterImageInterface $file
     * @return RasterImageProxy
     * @throws UnsupportedRasterImageTypeException
     */
    public static function createFromFile(FileInterface $file)
    {
        if (!$file instanceof RasterImageInterface) { // @phpstan-ignore instanceof.alwaysTrue
            throw new UnsupportedRasterImageTypeException('File must implement RasterImageInterface to create RasterImageProxy from.');
        }
        $dbFile = new RasterImageProxy(
            $file->getFileName(),
            $file->getFileSize(),
            $file->getMimeType(),
            $file->getTimestamp(),
            $file->getPixelWidth(),
            $file->getPixelHeight()
        );
        return $dbFile;
    }

    public function getPixelWidth(): int
    {
        return $this->pixelWidth;
    }

    public function getPixelHeight(): int
    {
        return $this->pixelHeight;
    }

    protected function reloadMetadata(FilesystemInterface $filesystem, string $path)
    {
        parent::reloadMetadata($filesystem, $path);

        $metadata = $filesystem->getMetadata($path);
        if (
            array_key_exists('width', $metadata)
            && array_key_exists('height', $metadata)
        ) {
            $this->pixelWidth = $metadata['width'];
            $this->pixelHeight = $metadata['height'];
        } else {
            $content = $filesystem->read($path);
            if ($content === false) {
                throw new Exception("Reading of file '{$path}' faild.");
            }
            $imageSize = getimagesizefromstring($content);
            if ($imageSize === false) {
                throw new UnsupportedRasterImageTypeException('getimagesizefromstring cannot determine size of image (fileUID=' . $this->getFileUID() . ';originUID=' . $this->getOriginUID() . '). If file type is WebP use php7.4');
            }
            $this->pixelWidth = $imageSize[0];
            $this->pixelHeight = $imageSize[1];
            $exif = @exif_read_data($filesystem->readStream($path)) ?: [];
            if (!empty($exif['Orientation']) && in_array($exif['Orientation'], [6, 8])) {
                $this->pixelWidth = $imageSize[1];
                $this->pixelHeight = $imageSize[0];
            }
        }
    }
}
