<?php

namespace IZON\DB\Files;

use IZON\DB\Files\Exceptions\DBFileException;
use IZON\DB\Files\Exceptions\NotHaveFileException;
use IZON\DB\Files\Exceptions\UnwritableLocationException;
use IZON\DB\Files\Helpers\EntityWrapperInterface;
use IZON\DB\Files\Helpers\PropertyWrapper;
use IZON\DB\Files\Proxies\FileProxy;
use IZON\DB\Files\Proxies\RasterImageProxy;
use IZON\DB\Files\Types\RasterImageType;
use IZON\IO\FileInterface;
use League\Flysystem\FileNotFoundException;
use League\Flysystem\FilesystemInterface;

/**
 * processor that processes entities with file properties
 */
class FilePropertyEntityProcessor
{
    /**
     * @return FilesystemInterface
     */
    protected $tmpFileSystem;

    /**
     * @return FilesystemInterface
     */
    protected $publicFileSystem;

    /**
     * @return FilesystemInterface
     */
    protected $privateFileSystem;

    /**
     * @var array
     */
    protected $fileSaveQueue = [];

    /**
     * @var array
     */
    protected $oldDeleteQueue = [];

    /**
     * @return FilesystemInterface
     */
    protected $noImageFileSystem;

    /**
     * @var string path for noimage in $noImageFileSystem
     */
    protected $noImagePath = '/images/noimage.jpg';


    public function __construct(
        FilesystemInterface $tmpSystem,
        FilesystemInterface $publicSystem,
        FilesystemInterface $privateSystem,
        FilesystemInterface $noImageFileSystem
    ) {
        $this->tmpFileSystem = $tmpSystem;
        $this->publicFileSystem = $publicSystem;
        $this->privateFileSystem = $privateSystem;
        $this->noImageFileSystem = $noImageFileSystem;
        $this->fileSaveQueue = [];
    }

    public function __destruct()
    {
        foreach ($this->fileSaveQueue as $tmpFileName) {
            $this->tmpFileSystem->delete($tmpFileName);
        }
    }

    // -------------------------------------------------------------------------
    // LOAD
    // -------------------------------------------------------------------------

    /**
     *
     * @param EntityWrapperInterface $entityWrapper
     */
    public function postLoad(EntityWrapperInterface $entityWrapper)
    {
        if (!$entityWrapper->hasAnyFileProperty()) {
            return;
        }

        $fileProperties = $entityWrapper->getFileProperties();
        foreach ($fileProperties as $property) {
            $value = $property->getValue();
            if (is_null($value)) {
                continue;
            }
            if (!$value instanceof FileProxy) {
                throw new DBFileException(
                    "FileProxy not set to property "
                    . "'{$property->getName()}' in class "
                    . "'{$entityWrapper->getEntityClass()}'."
                );
            }
            $path = $property->getGeneratedFileName($value);

            if ($property->isWebVisibleFile()) {
                $filesystem = $this->publicFileSystem;
            } else {
                $filesystem = $this->privateFileSystem;
            }

            if ($filesystem->has($path)) { // if file on target file system
                // inject it to proxy
                $value->injectSource($filesystem, $path);
            } else {
                // set noimage instead
                $value->injectSource($this->noImageFileSystem, $this->noImagePath);
            }
        }
    }

    // -------------------------------------------------------------------------
    // PERSIST
    // -------------------------------------------------------------------------

    public function prePersist(EntityWrapperInterface $entityWrapper)
    {
        if (!$entityWrapper->hasAnyFileProperty()) {
            return;
        }

        $fileProperties = $entityWrapper->getFileProperties();
        foreach ($fileProperties as $property) {
            $value = $property->getValue();

            $this->checkValue($property, $entityWrapper);

            if ($property->isNullValue()) {
                continue;
            }

            $tmpPath = $property->getTemporaryFileName();
            $status = $this->tmpFileSystem->putStream($tmpPath, $value->getFileStream());
            if (!$status) {
                throw new UnwritableLocationException('FILE COULD NOT BE SAVED IN TMP');
            }
            $this->fileSaveQueue[$tmpPath] = $tmpPath;
        }
    }

    public function postPersist(EntityWrapperInterface $entityWrapper)
    {
        if (!$entityWrapper->hasAnyFileProperty()) {
            return;
        }

        $fileProperties = $entityWrapper->getFileProperties();
        foreach ($fileProperties as $property) {
            if ($property->isNullValue()) {
                continue;
            }

            $value = $property->getValue();

            $tmpPath = $property->getTemporaryFileName();
            if (!array_key_exists($tmpPath, $this->fileSaveQueue)) {
                throw new DBFileException("File '{$tmpPath}' not found in TMP queue.");
            }

            $hasTmp = $this->tmpFileSystem->has($tmpPath);
            if (!$hasTmp) {
                throw new DBFileException("File '{$tmpPath}' not found in temporary location.");
            }

            if ($property->isWebVisibleFile()) {
                $filesystem = $this->publicFileSystem;
            } else {
                $filesystem = $this->privateFileSystem;
            }

            $resource = $this->tmpFileSystem->readStream($tmpPath);
            $path = $property->getGeneratedFileName(null, true);

            $status = $filesystem->putStream($path, $resource);
            if (!$status) {
                throw new UnwritableLocationException(
                    "File '{$path}' could not be saved in "
                    . ($property->isWebVisibleFile() ? 'public' : 'private')
                    . " location."
                );
            }

            // replace with moved file
            $proxyClass = FileProxy::class;
            if ($property->getType() == RasterImageType::RASTER_IMAGE) {
                $proxyClass = RasterImageProxy::class;
            }
            $fileProxy = $proxyClass::createFromFile($value);
            $fileProxy->injectSource($filesystem, $path);
            $property->setValue($fileProxy);
        }
    }

    // -------------------------------------------------------------------------
    // UPDATE
    // -------------------------------------------------------------------------

    /**
     *
     * @param EntityWrapperInterface $entityWrapper
     * @param array $oldValues
     * @throws UnwritableLocationException
     */
    public function preUpdate(EntityWrapperInterface $entityWrapper, array $oldValues)
    {
        if (!$entityWrapper->hasAnyFileProperty()) {
            return;
        }
        $recompute = false;
        $changedPropertyNames = array_keys($oldValues);
        foreach ($changedPropertyNames as $propertyName) {
            if (!$entityWrapper->isFileProperty($propertyName)) {
                continue;
            }

            /** @var PropertyWrapper $property */
            $property = $entityWrapper->getFileProperty($propertyName);
            $value = $property->getValue();
            $oldValue = $oldValues[$property->getName()];

            if (
                $value !== null // new value is not null
                && $value->getFileUID() == $this->noImagePath
                && $value->getOriginUID() === $this->noImageFileSystem->getFilesystemUID()
            ) {
                $property->setValue($oldValue);
                $recompute = true;
            }

            $this->checkValue($property, $entityWrapper);


            if ($value === null) {
                // new value is null, do not process file
                continue;
            }

            if (
                $oldValue instanceof FileInterface
                && $value->getOriginUID() == $oldValue->getOriginUID()
                && $value->getFileUID() == $oldValue->getFileUID()
            ) {
                continue; // points to same file, do not move file
            }


            $tmpPath = $property->getTemporaryFileName();
            $status = $this->tmpFileSystem->putStream($tmpPath, $value->getFileStream());
            if (!$status) {
                throw new UnwritableLocationException('FILE COULD NOT BE SAVED IN TMP');
            }
            $this->fileSaveQueue[$tmpPath] = $tmpPath;

            $oldValue = $oldValues[$property->getName()];
            $oldPath = $property->getGeneratedFileName($oldValue);
            $this->markOldFileDelete(
                $entityWrapper->getEntityClass(),
                $entityWrapper->getUID(),
                $property->getName(),
                $oldPath
            );
        }
        if ($recompute) {
            $entityWrapper->recomputeChangeSet();
        }
    }

    /**
     *
     * @param EntityWrapperInterface $entityWrapper
     * @throws DBFileException
     * @throws UnwritableLocationException
     * @throws FileNotFoundException
     */
    public function postUpdate(EntityWrapperInterface $entityWrapper)
    {
        if (!$entityWrapper->hasAnyFileProperty()) {
            return;
        }

        /** @var PropertyWrapper[] $fileProperties */
        $fileProperties = $entityWrapper->getFileProperties();
        foreach ($fileProperties as $property) {
            if ($property->isWebVisibleFile()) {
                $filesystem = $this->publicFileSystem;
            } else {
                $filesystem = $this->privateFileSystem;
            }

            if ($property->isNullValue()) { // property value is null
                // delete old files if any exists
                $oldFiles = $filesystem->listContents($property->getGeneratedFileDir());
                foreach ($oldFiles as $oldFile) {
                    $filesystem->delete($oldFile['path']);
                }
                // no need to process file any further
                continue;
            }

            $value = $property->getValue();

            $tmpPath = $property->getTemporaryFileName();
            if (!array_key_exists($tmpPath, $this->fileSaveQueue)) { // file for this property was not changed
                // no need to process file as it is already in target filesystem
                continue;
            }

            $hasTmp = $this->tmpFileSystem->has($tmpPath);
            if (!$hasTmp) {
                throw new DBFileException("File '{$tmpPath}' not found in temporary location.");
            }

            $resource = $this->tmpFileSystem->readStream($tmpPath);
            $path = $property->getGeneratedFileName(null, true);
            $status = $filesystem->putStream($path, $resource);
            if (!$status) {
                throw new UnwritableLocationException(
                    "File '{$path}' could not be saved in "
                    . ($property->isWebVisibleFile() ? 'public' : 'private')
                    . " location."
                );
            }

            // replace with moved file
            $proxyClass = FileProxy::class;
            if ($property->getType() == RasterImageType::RASTER_IMAGE) {
                $proxyClass = RasterImageProxy::class;
            }
            $fileProxy = $proxyClass::createFromFile($value);
            $fileProxy->injectSource($filesystem, $path);
            $property->setValue($fileProxy);

            // delete from tmp file system
            //            $this->tmpFileSystem->delete($tmpPath);
            //            $this->fileSaveQueue = array_filter($this->fileSaveQueue, function($item) use($tmpPath) {
            //                return $item !== $tmpPath;
            //            });

            $oldPath = $this->unmarkOldFileDelete($entityWrapper->getEntityClass(), $entityWrapper->getUID(), $property->getName());
            if ($filesystem->has($oldPath) && $oldPath != $path) {
                $filesystem->delete($oldPath);
            }
        }
    }

    // -------------------------------------------------------------------------
    // REMOVE
    // -------------------------------------------------------------------------

    /**
     *
     * @param EntityWrapperInterface $entityWrapper
     */
    public function postRemove(EntityWrapperInterface $entityWrapper)
    {
        if (!$entityWrapper->hasAnyFileProperty()) {
            return;
        }

        $fileProperties = $entityWrapper->getFileProperties();
        foreach ($fileProperties as $property) {
            if ($property->isNullValue()) {
                continue;
            }
            $value = $property->getValue();
            $path = $value->getFileUID();
            if ($property->isWebVisibleFile()) {
                $filesystem = $this->publicFileSystem;
            } else {
                $filesystem = $this->privateFileSystem;
            }

            // remove files in directory storing property files
            $dirname = dirname($path);
            if ($filesystem->has($dirname)) { // has files dirname
                $files = $filesystem->listContents($dirname);
                foreach ($files as $file) {
                    $filesystem->delete($file['path']);
                }
            }
            // remove empty subdirs on the path to root dir, do not remove root) dir
            while (!empty($dirname) && $dirname != '/') {
                $list = $filesystem->listContents($dirname);
                if (!empty($list)) {
                    break;
                }
                $filesystem->deleteDir($dirname);
                $dirname = dirname($dirname);
            }
        }
    }

    public function clearFileQueues()
    {
        foreach ($this->fileSaveQueue as $tmpFilePathHash => $tmpFilePath) {
            $this->tmpFileSystem->delete($tmpFilePath);
            unset($this->fileSaveQueue[$tmpFilePathHash]);
        }
    }

    /**
     * sets path to no image on $noImageFileSystem
     * @param string $noImagePath
     */
    public function setNoImagePath(string $noImagePath): void
    {
        $this->noImagePath = $noImagePath;
    }

    // -------------------------------------------------------------------------
    // GENERATORS
    // -------------------------------------------------------------------------

    /**
     *
     * @param PropertyWrapper $property
     * @param EntityWrapperInterface $entityWrapper
     * @throws NotHaveFileException
     * @throws DBFileException
     */
    protected function checkValue(PropertyWrapper $property, EntityWrapperInterface $entityWrapper)
    {
        $value = $property->getValue();
        if (!$value instanceof FileInterface && !$property->isNullable()) {
            throw new NotHaveFileException(
                "File not set to property "
                . "'{$property->getName()}' in class "
                . "'{$entityWrapper->getEntityClass()}'."
            );
        } elseif (!$value instanceof FileInterface && !$property->isNullValue()) {
            throw new DBFileException(
                "File or null not set to property "
                . "'{$property->getName()}' in class "
                . "'{$entityWrapper->getEntityClass()}'."
            );
        }
    }

    protected function markOldFileDelete($class, $uid, $field, $path)
    {
        if (!isset($this->oldDeleteQueue[$class . '_' . $uid])) {
            $this->oldDeleteQueue[$class . '_' . $uid] = [];
        }
        $this->oldDeleteQueue[$class . '_' . $uid][$field] = $path;
    }

    protected function unmarkOldFileDelete($class, $uid, $field)
    {
        $value = $this->oldDeleteQueue[$class . '_' . $uid][$field];
        unset($this->oldDeleteQueue[$class . '_' . $uid][$field]);
        if (empty($this->oldDeleteQueue[$class . '_' . $uid])) {
            unset($this->oldDeleteQueue[$class . '_' . $uid]);
        }
        return $value;
    }
}
