<?php

namespace IZON\DB\Files\Types;

use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Types\ConversionException;
use Doctrine\DBAL\Types\Type;
use IZON\DB\Exceptions\DBException;
use IZON\DB\Files\Helpers\FileUtils;
use IZON\DB\Files\Proxies\FileProxy;
use IZON\IO\FileInterface;

/**
 * File type for mapping to db
 */
class FileType extends Type {

    const FILE = 'file';
    
    public function getName(): string {
        return self::FILE;
    }

    public function getSQLDeclaration(array $fieldDeclaration, AbstractPlatform $platform): string {
        return $platform->getJsonTypeDeclarationSQL($fieldDeclaration);
    }

    public function convertToPHPValue($value, AbstractPlatform $platform) {
        // This is executed when the value is read from the database. Make your conversions here, optionally using the $platform.
        if( $value === null ) {
            return null;
        }
        
        $unserializedArray = json_decode($value, true);
        if(json_last_error() !== JSON_ERROR_NONE) {
            throw ConversionException::conversionFailedSerialization($value, 'json', json_last_error_msg());
        }
        
        // DOTO: how to handle that db value does not contain some necessary data ???
        
        $file = new FileProxy(
                $unserializedArray['fileName'], 
                $unserializedArray['fileSize'], 
                $unserializedArray['mimeType'],
                $unserializedArray['timestamp']
            );
        
        return $file;
    }

    public function convertToDatabaseValue($value, AbstractPlatform $platform) {
        // This is executed when the value is written to the database. Make your conversions here, optionally using the $platform.
        if( $value === null ) {
            return null;
        }
        
        if( ! $value instanceof FileInterface ) {
            throw new DBException('Unsupported file class '. get_class($value));
        }
        
        $serializedArray = [
            'fileName' => FileUtils::getSafeFileName($value->getFileName()),
            'fileSize' => $value->getFileSize(),
            'mimeType' => $value->getMimeType(),
            'timestamp' => $value->getTimestamp(),
        ];
        
        return json_encode($serializedArray);
    }

}
