<?php

namespace IZON\DB\Files\Helpers;

use Doctrine\ORM\Mapping\ClassMetadataInfo;
use Doctrine\ORM\Mapping\Column;
//use Doctrine\ORM\Mapping\Driver\AttributeReader;
use IZON\DB\Files\Exceptions\DBFileException;
use IZON\DB\Files\Mapping\File;
use IZON\DB\Files\Mapping\Image;
use IZON\DB\Files\Types\FileType;
use IZON\DB\Files\Types\RasterImageType;
use IZON\IO\FileInterface;
use ReflectionProperty;

/**
 * Description of PropertyWrapper
 *
 * @author Lukas Linhart
 */
class PropertyWrapper {
    // -------------------------------------------------------------------------
    // EXTERNALLY SET PROPERTIES
    // -------------------------------------------------------------------------

    /**
     * @var string
     */
    protected $propertyName;

    /**
     * @var string conlum data type a.i. RasterImageType::RASTER_IMAGE of  FileType::FILE
     */
    protected $type;

    /**
     * @var EntityWrapper
     */
    protected $parentEntity;

    /**
     *
     * @var ClassMetadataInfo
     */
    protected $metadata;

    // -------------------------------------------------------------------------
    // CONSTRUCTOR
    // -------------------------------------------------------------------------

    /**
     *
     * @param string $propertyName
     * @param string $type
     * @param EntityWrapper $parentEntity
     * @param ClassMetadataInfo $metadata
     */
    public function __construct(
        string $propertyName,
        string $type,
        EntityWrapper $parentEntity,
        ClassMetadataInfo $metadata
    ) {
        $this->propertyName = $propertyName;
        $this->type = $type;
        $this->parentEntity = $parentEntity;
        $this->metadata = $metadata;
    }

    // -------------------------------------------------------------------------
    // PUBLIC METHODS
    // -------------------------------------------------------------------------

    /**
     *
     * @return string
     */
    public function getName(): string {
        return $this->propertyName;
    }

    /**
     *
     * @return FileInterface|null
     */
    public function getValue() {
        return $this->metadata->getFieldValue($this->parentEntity->getEntity(), $this->propertyName);
    }

    /**
     *
     * @param FileInterface|null $file
     */
    public function setValue(?FileInterface $file) {
        $this->metadata->setFieldValue($this->parentEntity->getEntity(), $this->propertyName, $file);
    }

    /**
     *
     * @return bool
     */
    public function isNullValue(): bool {
        return is_null($this->getValue());
    }

    /**
     *
     * @return bool
     * @throws DBFileException
     */
    public function isNullable(): bool {
        return $this->metadata->isNullable($this->propertyName);
    }

    /**
     * @return bool
     */
    public function isWebVisibleFile(): bool {
        $fieldMapping = $this->metadata->getFieldMapping($this->propertyName);
        if(
            array_key_exists('options', $fieldMapping)
            && is_array($fieldMapping['options'])
            && array_key_exists('webVisible', $fieldMapping['options'])
        ) {
            return $fieldMapping['options']['webVisible'];
        }
        return true;
    }

    /**
     *
     * @return string
     * @throws DBFileException
     */
    public function getGeneratedFileDir() {
        $uid = $this->parentEntity->getUID();
        $tableName = $this->convertUnderscoreToKebab($this->parentEntity->getTableName());
        return '/' . $tableName . '/' . $uid . '/' . $this->convertCamelToKebab($this->propertyName);
    }

    /**
     *
     * @return string
     * @throws DBFileException
     */
    public function getGeneratedFileName(FileInterface $value = null, bool $saveFileName = false) {
        if(empty($value)) {
            $value = $this->getValue();
        }

        if(is_null($value)) {
            $uid = $this->parentEntity->getUID();
            throw new DBFileException(
                "Could not generate name for entity "
                . "'{$this->parentEntity->getEntityClass()}' with id '{$uid}' "
                . "becouse property {$this->propertyName} "
                . "does not have any file."
            );
        }

        $fileName = $value->getFileName();
        if($saveFileName) {
            $fileName = FileUtils::getSafeFileName($fileName);
        }
        return $this->getGeneratedFileDir() . '/' . $fileName;
    }

    /**
     *
     * @return string
     * @throws DBFileException
     */
    public function getTemporaryFileName(): string {
        $value = $this->getValue();
        $tableName = $this->convertUnderscoreToKebab($this->parentEntity->getTableName());
        if(is_null($value)) {
            throw new DBFileException(
                "Could not generate temporary name for entity "
                . "'{$this->parentEntity->getEntityClass()}' ' "
                . "becouse property {$this->propertyName} "
                . "does not have any file."
            );
        }
        $timestamp = $value->getTimestamp();
        $fileName = FileUtils::getSafeFileName($value->getFileName());
        $fileSize = $value->getFileSize();
        return '/' . $tableName . '/' . $this->convertCamelToKebab($this->propertyName) . '/t' . $timestamp .'-size'. $fileSize .'-'. $fileName;
    }

    /**
     * @return string returnd type of propety RasterImageType::RASTER_IMAGE of  FileType::FILE
     */
    public function getType(): string {
        return $this->type;
    }
    
    /**
     * converts camel case to kebab case
     * @param string $input
     * @return string
     */
    protected function convertCamelToKebab(string $input) {
        $string = preg_replace('/(?<=[a-z])([A-Z])/', '-$1', $input);
        return strtolower($string);
    }

    /**
     * conversts underscore to kebam case
     * @param string $input
     * @return string|string[]
     */
    protected function convertUnderscoreToKebab(string $input) {
        return str_replace('_', '-', $input);
    }
}
