<?php

namespace IZON\DB\Files\Types;

use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Types\ConversionException;
use Doctrine\DBAL\Types\Type;
use IZON\DB\Exceptions\DBException;
use IZON\DB\Files\Helpers\FileUtils;
use IZON\DB\Files\Proxies\FileProxy;
use IZON\IO\FileInterface;

/**
 * RasterImage type for mapping to db
 */
class RasterImageType extends Type {

    const RASTER_IMAGE = 'raster_image';

    public function getName(): string {
        return self::RASTER_IMAGE;
    }

    public function getSQLDeclaration(array $fieldDeclaration, AbstractPlatform $platform): string {
        return $platform->getJsonTypeDeclarationSQL($fieldDeclaration);
    }

    public function convertToPHPValue($value, AbstractPlatform $platform) {
        // This is executed when the value is read from the database. Make your conversions here, optionally using the $platform.
        if( $value === null ) {
            return null;
        }

        $unserializedArray = json_decode($value, true);
        if(json_last_error() !== JSON_ERROR_NONE) {
            throw ConversionException::conversionFailedSerialization($value, 'json', json_last_error_msg());
        }

        // DOTO: how to handle that db value does bnot contain some necessary data ???

        $file = new \IZON\DB\Files\Proxies\RasterImageProxy(
            $unserializedArray['fileName'],
            $unserializedArray['fileSize'],
            $unserializedArray['mimeType'],
            $unserializedArray['timestamp'],
            $unserializedArray['pixelWidth'],
            $unserializedArray['pixelHeight']
        );

        return $file;
    }

    public function convertToDatabaseValue($value, AbstractPlatform $platform) {
        // This is executed when the value is written to the database. Make your conversions here, optionally using the $platform.
        if( $value === null ) {
            return null;
        }

        if( ! $value instanceof \IZON\IO\RasterImageInterface ) {
            throw new DBException('Unsupported file class '. get_class($value));
        }

        $serializedArray = [
            'fileName' => FileUtils::getSafeFileName($value->getFileName()),
            'fileSize' => $value->getFileSize(),
            'mimeType' => $value->getMimeType(),
            'timestamp' => $value->getTimestamp(),
            'pixelWidth' => $value->getPixelWidth(),
            'pixelHeight' => $value->getPixelHeight(),
        ];

        return json_encode($serializedArray);
    }

}
