<?php

/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

namespace IZON\DB\Files\Helpers;

use Doctrine\Common\Annotations\AnnotationReader;
use Doctrine\ORM\Mapping\Column;
use IZON\DB\Files\Exceptions\DBFileException;
use IZON\DB\Files\Mapping\File;
use IZON\DB\Files\Mapping\Image;
use IZON\DB\Files\Types\FileType;
use IZON\DB\Files\Types\RasterImageType;
use IZON\IO\FileInterface;
use ReflectionProperty;

/**
 * Description of PropertyWrapper
 *
 * @author Lukas Linhart
 */
class PropertyWrapper {
    // -------------------------------------------------------------------------
    // EXTERNALLY SET PROPERTIES
    // -------------------------------------------------------------------------

    /**
     * @var string
     */
    protected $propertyName;

    /**
     * @var string conlum data type a.i. RasterImageType::RASTER_IMAGE of  FileType::FILE
     */
    protected $type;

    /**
     * @var EntityWrapper
     */
    protected $parentEntity;

    /**
     * @var AnnotationReader
     */
    protected $annotationReader;

    // -------------------------------------------------------------------------
    // INTERNAL HELPING PROPERTIES
    // -------------------------------------------------------------------------

    /**
     *
     * @var ReflectionProperty
     */
    protected $reflectionProperty;

    // -------------------------------------------------------------------------
    // CONSTRUCTOR
    // -------------------------------------------------------------------------

    /**
     *
     * @param string $propertyName
     * @param string $type
     * @param EntityWrapper $parentEntity
     * @param AnnotationReader $reader
     */
    public function __construct(
        string $propertyName,
        string $type,
        EntityWrapper $parentEntity,
        AnnotationReader $reader
    ) {
        $this->propertyName = $propertyName;
        $this->type = $type;
        $this->parentEntity = $parentEntity;
        $this->annotationReader = $reader;
    }

    // -------------------------------------------------------------------------
    // PUBLIC METHODS
    // -------------------------------------------------------------------------

    /**
     *
     * @return string
     */
    public function getName(): string {
        return $this->propertyName;
    }

    /**
     *
     * @return FileInterface|null
     */
    public function getValue() {
        $this->initReflectionProperty();
        $this->reflectionProperty->setAccessible(true);
        return $this->reflectionProperty->getValue($this->parentEntity->getEntity());
    }

    /**
     *
     * @param FileInterface|null $file
     */
    public function setValue(?FileInterface $file) {
        $this->initReflectionProperty();
        $this->reflectionProperty->setAccessible(true);
        $this->reflectionProperty->setValue($this->parentEntity->getEntity(), $file);
    }

    /**
     *
     * @return bool
     */
    public function isNullValue(): bool {
        return is_null($this->getValue());
    }

    /**
     *
     * @return bool
     * @throws DBFileException
     */
    public function isNullable(): bool {
        $annotation = $this->getAnnotation(Column::class);
        if (!$annotation instanceof Column) {
            throw new DBFileException(
                "Property '{$this->propertyName}' in class "
                . "'{$this->parentEntity->getEntityClass()}' not has annotation "
                . "\Doctrine\ORM\Mapping\Column::class."
            );
        }
        /* @var $annotation Column */
        return $annotation->nullable;
    }

    /**
     *
     * @return bool
     */
    public function isPublicFile(): bool {
        $annotation = $this->getAnnotation(File::class);
        if (!$annotation instanceof File
            && !$annotation instanceof Image) {
            return true;
        }
        /** @var File $annotation */
        return $annotation->webVisible;
    }

    /**
     *
     * @return string
     * @throws DBFileException
     */
    public function getGeneratedFileDir() {
        $uid = $this->parentEntity->getUID();
        $tableName = $this->convertUnderscoreToKebab($this->parentEntity->getTableName());
        return '/' . $tableName . '/' . $uid . '/' . $this->convertCamelToKebab($this->propertyName);
    }

    /**
     *
     * @return string
     * @throws DBFileException
     */
    public function getGeneratedFileName(FileInterface $value = null) {
        if(empty($value)) {
            $value = $this->getValue();
        }

        if(is_null($value)) {
            $uid = $this->parentEntity->getUID();
            throw new DBFileException(
                "Could not generate name for entity "
                . "'{$this->parentEntity->getEntityClass()}' with id '{$uid}' "
                . "becouse property {$this->propertyName} "
                . "does not have any file."
            );
        }

        $fileName = $value->getFileName();

        return $this->getGeneratedFileDir() . '/' . $fileName;
    }

    /**
     *
     * @return string
     * @throws DBFileException
     */
    public function getTemporaryFileName(): string {
        $value = $this->getValue();
        $tableName = $this->convertUnderscoreToKebab($this->parentEntity->getTableName());
        if(is_null($value)) {
            throw new DBFileException(
                "Could not generate temporary name for entity "
                . "'{$this->parentEntity->getEntityClass()}' ' "
                . "becouse property {$this->propertyName} "
                . "does not have any file."
            );
        }
        $timestamp = $value->getTimestamp();
        $fileName = $value->getFileName();
        return '/' . $tableName . '/' . $this->convertCamelToKebab($this->propertyName) . '/' . $timestamp . '-' . $fileName;
    }

    /**
     * get anotation by class/name
     * @param string $annotationClass
     * @return object|null
     */
    public function getAnnotation(string $annotationClass) {
        $this->initReflectionProperty();
        return $this->annotationReader->getPropertyAnnotation($this->reflectionProperty, $annotationClass);
    }

    /**
     * @return string returnd type of propety RasterImageType::RASTER_IMAGE of  FileType::FILE
     */
    public function getType(): string {
        return $this->type;
    }

    // -------------------------------------------------------------------------
    // INTERNAL METHODS
    // -------------------------------------------------------------------------

    protected function initReflectionProperty() {
        if (!$this->reflectionProperty instanceof \ReflectionProperty) {
            $this->reflectionProperty = $this->parentEntity
                ->getReflectionClass()
                ->getProperty($this->propertyName);
        }
    }

    /**
     * converts camel case to kebab case
     * @param string $input
     * @return string
     */
    protected function convertCamelToKebab(string $input) {
        $string = preg_replace('/(?<=[a-z])([A-Z])/', '-$1', $input);
        return strtolower($string);
    }

    /**
     * conversts underscore to kebam case
     * @param string $input
     * @return string|string[]
     */
    protected function convertUnderscoreToKebab(string $input) {
        return str_replace('_', '-', $input);
    }
}
