<?php

namespace IZON\DB\Files\Helpers;

use Doctrine\Common\Annotations\AnnotationReader;
use Doctrine\Common\Inflector\Inflector;
use Doctrine\Common\Persistence\ObjectManager;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use Doctrine\ORM\Mapping\Id;
use IZON\DB\Files\Exceptions\DBFileException;
use IZON\DB\Files\Exceptions\NotPersistedEntityException;
use IZON\DB\Files\Types\FileType;
use IZON\DB\Files\Types\RasterImageType;
use ReflectionClass;

/**
 * Description of EntityWrapper
 *
 * @author Lukáš Linhart
 */
class EntityWrapper implements EntityWrapperInterface {
    // -------------------------------------------------------------------------
    // EXTERNALLY SET PROPERTIES
    // -------------------------------------------------------------------------

    /**
     *
     * @var object
     */
    protected $entity;

    /**
     *
     * @var ObjectManager
     */
    protected $manager;

    /**
     *
     * @var bool
     */
    protected $isPersisted;

    // -------------------------------------------------------------------------
    // INTERNAL HELPING PROPERTIES
    // -------------------------------------------------------------------------

    /**
     *
     * @var \Doctrine\Common\Annotations\AnnotationReader;
     */
    protected $annotationReader;

    /**
     *
     * @var string
     */
    protected $entityClass;

    /**
     *
     * @var ClassMetadataInfo
     */
    protected $metadata;

    /**
     *
     * @var string|null
     */
    protected $uid = null;

    /**
     *
     * @var ReflectionClass
     */
    protected $reflectionClass;

    /**
     *
     * @var string[] array of property names
     */
    protected $entityFields;

    /**
     *
     * @var PropertyWrapper[]|null
     */
    protected $fileProperties;

    // -------------------------------------------------------------------------
    // CONSTRUCTOR
    // -------------------------------------------------------------------------

    /**
     *
     * @param object $entity
     * @param ObjectManager $manager
     * @param bool $isPersisted default true
     */
    public function __construct($entity, ObjectManager $manager, bool $isPersisted = true) {
        $this->entity = $entity;
        $this->manager = $manager;
        $this->isPersisted = $isPersisted;

        $this->entityClass = \get_class($entity);
        $metadata = $this->manager->getClassMetadata($this->entityClass);
        if ($metadata instanceof ClassMetadataInfo) {
            $this->metadata = $metadata;
        } else {
            throw new DBFileException(
                "Metadata for entity of class {$this->entityClass} "
                . "is not instance of '" . ClassMetadataInfo::class . "' "
                . "but '" . \get_class($metadata) . "'"
            );
        }
        $this->reflectionClass = $this->metadata->getReflectionClass();
        $this->entityFields = $this->metadata->getFieldNames();
        $this->annotationReader = new AnnotationReader();
    }

    // -------------------------------------------------------------------------
    // PUBLIC METHODS
    // -------------------------------------------------------------------------

    /**
     * full class name with namespace
     * @return string
     */
    public function getEntityClass(): string {
        return $this->entityClass;
    }

    /**
     *
     * @return object
     */
    public function getEntity() {
        return $this->entity;
    }

    /**
     *
     * @return ReflectionClass
     */
    public function getReflectionClass(): \ReflectionClass {
        return $this->reflectionClass;
    }

    /**
     * unique ident
     * @return string
     * @throws NotPersistedEntityException if entity is not persisted
     */
    public function getUID(): string {
        if (!$this->isPersisted) {
            throw new NotPersistedEntityException();
        }
        if (\is_null($this->uid)) {
            $this->uid = $this->generateUid();
        }
        return $this->uid;
    }

    /**
     *
     * @return PropertyWrapper[]
     */
    public function getFileProperties(): array {
        if (is_null($this->fileProperties)) { // not initialized yet
            $this->fileProperties = [];
            foreach ($this->entityFields as $field) {
                $type = $this->metadata->getTypeOfField($field);
                if($type != FileType::FILE &&
                    $type != RasterImageType::RASTER_IMAGE) {
                    continue;
                }
                $this->fileProperties[$field] = new PropertyWrapper($field, $type, $this, $this->annotationReader);
            }
        }
        return $this->fileProperties;
    }

    /**
     *
     * @param string $name
     * @return bool
     */
    public function isFileProperty(string $name): bool {
        $properties = $this->getFileProperties();
        return \array_key_exists($name, $properties);
    }

    /**
     *
     * @param string $name
     * @return PropertyWrapper
     * @throws DBFileException
     */
    public function getFileProperty(string $name): PropertyWrapper {
        $properties = $this->getFileProperties();
        if (!isset($properties[$name])) {
            throw new DBFileException(
                "Property '$name' is not file property "
                . "in class '{$this->entityClass}'"
            );
        }
        return $properties[$name];
    }

    /**
     *
     * @return bool
     */
    public function hasAnyFileProperty(): bool {
        $properties = $this->getFileProperties();
        return !empty($properties);
    }

    /**
     * return table name like "table_name"
     * @return string
     */
    public function getTableName(): string {
        return $this->metadata->getTableName();
    }

    /**
     * return table name like "tableName"
     * @return string
     */
    public function getCamelizeTableName(): string {
        return Inflector::camelize($this->getTableName());
    }

    // -------------------------------------------------------------------------
    // PROTECTED METHODS
    // -------------------------------------------------------------------------

    protected function generateUid(): string {
        $ids = [];
        foreach ($this->entityFields as $field) {
            $reflectionProperty = $this->reflectionClass->getProperty($field);
            $anotations = $this->annotationReader->getPropertyAnnotations($reflectionProperty);
            foreach ($anotations as $anotation) {
                if (!$anotation instanceof Id) {
                    continue;
                }
                $reflectionProperty->setAccessible(true);
                $ids[] = $reflectionProperty->getValue($this->entity);
            }
        }
        return implode('-', $ids);
    }

}
