<?php

namespace IZON\Date;

use DateTimeImmutable;
use DateTimeInterface;
use DateTimeZone;

class DateTimeImmutableFactory
{
    public static function now(): DateTimeImmutable
    {
        return new DateTimeImmutable();
    }

    /**
     * @param DateTimeInterface $date day to create start of the day for
     * @return DateTimeImmutable dateTime representing start of the day 00:00:00 hours for same timezone as $date
     * @throws \Exception
     */
    public static function createStartOfDay(DateTimeInterface $date): DateTimeImmutable
    {
        return self::createFromFormat(
            "!Y-m-d H:i:s",
            $date->format('Y-m-d') . ' 00:00:00',
            $date->getTimezone()
        );
    }

    /**
     * @return DateTimeImmutable
     * @return DateTimeImmutable dateTime representing start of today 00:00:00 in system timezone
     * @throws \Exception
     */
    public static function createStartOfToday(): DateTimeImmutable
    {
        return self::createStartOfDay(new DateTimeImmutable());
    }

    /**
     * @param DateTimeInterface $date date string with format Y-m-d
     * @return DateTimeImmutable dateTime representing end of the day 23:59:59 hours
     * @throws \Exception
     */
    public static function createEndOfDay(DateTimeInterface $date): DateTimeImmutable
    {
        return self::createFromFormat(
            "!Y-m-d H:i:s",
            $date->format('Y-m-d') . ' 23:59:59',
            $date->getTimezone()
        );
    }

    /**
     * @return DateTimeImmutable
     * @return DateTimeImmutable dateTime representing start of today 00:00:00 in system timezone
     * @throws \Exception
     */
    public static function createEndOfToday(): DateTimeImmutable
    {
        return self::createEndOfDay(new DateTimeImmutable());
    }

    public static function createFromFormat(string $format, string $dateTime, ?DateTimeZone $timezone = null): DateTimeImmutable
    {
        if (!str_starts_with($format, '!') && !str_ends_with($format, '|')) { // always use strict format
            $format = '!' . $format;
        }
        $createdDateTime = DateTimeImmutable::createFromFormat($format, $dateTime, $timezone);
        if ($createdDateTime === false) {
            throw new \Exception('Unable to create DateTimeImmutable for ' . $dateTime . ' with format ' . $format . '.');
        }
        return $createdDateTime;
    }
}
