<?php

declare(strict_types=1);

namespace IZON\DataGrid\DataSource;

use IZON\DataGrid\Filter\FilterDate;
use IZON\DataGrid\Filter\FilterDateRange;
use IZON\DataGrid\Filter\FilterMultiSelect;
use IZON\DataGrid\Filter\FilterRange;
use IZON\DataGrid\Filter\FilterSelect;
use IZON\DataGrid\Filter\FilterText;
use IZON\DataGrid\Utils\PaginatorDataSource;
use IZON\DataGrid\Utils\Sorting;
use IZON\DB\Dao;
use IZON\DB\Paginator\PaginatorConfig;

/**
 * @method void onDataLoaded(array $result)
 */
class PaginatorConfigDataSource extends FilterableDataSource implements IDataSource {

    /**
     * Event called when datagrid data is loaded.
     *
     * @var array|callable[]
     */
    public $onDataLoaded;

    /** @var PaginatorDataSource */
    protected $dataSource;

    /** @var string */
    protected $primaryKey;

    /** @var PaginatorConfig */
    protected $config;

    /** @var Dao */
    protected $dao;


    public function __construct(PaginatorDataSource $dataSource, string $primaryKey) {
        $this->dataSource = $dataSource;
        $this->primaryKey = $primaryKey;

        $this->config = $dataSource->getConfig();
        $this->dao = $dataSource->getDao();
    }

    /********************************************************************************
     *                          IDataSource implementation                          *
     ********************************************************************************/

    public function getCount(): int {
        $config = clone $this->config;

        $config->setFirstResult(null);
        $config->setMaxPageSize(null);

        return $this->dao->paginate($config)->getTotalCount();
    }

    /**
     * {@inheritDoc}
     */
    public function getData(): array {
        return $this->dataSource->paginate($this->config)->getPage();
    }

    /**
     * {@inheritDoc}
     */
    public function filterOne(array $condition): IDataSource {
        die('not yet');

        foreach($condition as $column => $value) {
            $c = $this->checkAliases($column);

            $this->dataSource->andWhere("$c = :$p")
                ->setParameter($p, $value);
        }

        return $this;
    }

    public function limit(int $offset, int $limit): IDataSource {
        $this->config->setFirstResult($offset);
        $this->config->setMaxPageSize($limit);

        return $this;
    }

    public function sort(Sorting $sorting): IDataSource {
        die('not yet');
        if(is_callable($sorting->getSortCallback())) {
            call_user_func(
                $sorting->getSortCallback(),
                $this->dataSource,
                $sorting->getSort()
            );

            return $this;
        }

        $sort = $sorting->getSort();

        if($sort !== []) {
            foreach($sort as $column => $order) {
                $this->dataSource->addOrderBy($this->checkAliases((string)$column), $order);
            }
        } else {
            /**
             * Has the statement already a order by clause?
             */
            if(!(bool)$this->dataSource->getDQLPart('orderBy')) {
                $this->dataSource->orderBy($this->checkAliases($this->primaryKey));
            }
        }

        return $this;
    }

    protected function applyFilterDate(FilterDate $filter): void {
        // TODO: Implement applyFilterDate() method.
    }

    protected function applyFilterDateRange(FilterDateRange $filter): void {
        // TODO: Implement applyFilterDateRange() method.
    }

    protected function applyFilterRange(FilterRange $filter): void {
        // TODO: Implement applyFilterRange() method.
    }

    protected function applyFilterText(FilterText $filter): void {
        // TODO: Implement applyFilterText() method.
    }

    protected function applyFilterMultiSelect(FilterMultiSelect $filter): void {
        // TODO: Implement applyFilterMultiSelect() method.
    }

    protected function applyFilterSelect(FilterSelect $filter): void {
        // TODO: Implement applyFilterSelect() method.
    }


    protected function getDataSource() {
        return $this->dataSource;
    }

    public function filterIn(string $columng, array $inArray): IDataSource {
        // TODO: Implement filterIn() method.
    }

    public function updatePosition(
        string $property,
               $currentId,
        int    $currentPosition,
        ?int   $prevPosition,
        ?int   $nextPosition
    ): IDataSource {
        // TODO: Implement updatePosition() method.
    }
}
