<?php

declare(strict_types=1);

namespace IZON\DataGrid\Filter;

use IZON\Admin\MVC\ModuleInfo;
use IZON\DataGrid\DataGrid;
use IZON\Forms\Fields\BaseField;
use IZON\Forms\Form;
use function IZON\Admin\Forms\getAdminField;

abstract class AFilter {

    /** @var DataGrid|null */
    protected $dataGrid = null;

    /** @var ModuleInfo */
    protected $module;

    /** @var string */
    protected $key;

    /** @var string */
    protected $name;

    /** @var mixed */
    protected $value;

    /** @var string|null */
    private $placeholder;

    /** @var array */
    protected $attributes = ['class' => 'datagrid-reload-input'];

    /** @var BaseField */
    protected $baseField;

    /** @var string|null */
    protected $template;

    /** @var string|null */
    protected $type;

    /** @var bool */
    protected $valueSet = false;

    /** @var callable|null */
    protected $conditionCallback;

    public function __construct(string $key, string $name) {
        $this->key = $key;
        $this->name = $name;
    }

    // SETTER

    public function setDataGrid(DataGrid $dataGrid): void {
        if(!$this->dataGrid instanceof DataGrid) {
            $this->dataGrid = $dataGrid;
            $this->attributes['data-datagrid-param-name'] = $this->dataGrid->getName(sprintf('filter-%s', $this->key));
            $this->attributes['autocomplete'] = 'off';
        }
    }

    public function setModule(ModuleInfo $module): void {
        $this->module = $module;
    }

    public function setPlaceholder(string $placeholder): self {
        $this->placeholder = $placeholder;
        $this->setAttribute('placeholder', $this->placeholder);
        return $this;
    }

    public function addAttribute(string $name, $value): self {
        $this->attributes[$name][] = $value;
        return $this;
    }

    public function setAttribute(string $name, $value): self {
        $this->attributes[$name] = is_array($value) ? implode(' ', $value) : $value;

        return $this;
    }

    public function setValue($value): self {
        $this->value = $value;
        $this->valueSet = true;

        return $this;
    }


    abstract public function addToFormContainer(Form $form): void;

    // GETTER

    public function getAttributes(): array {
        return $this->attributes;
    }

    public function getPlaceholder(): ?string {
        return $this->placeholder;
    }

    public function getKey(): string {
        return $this->key;
    }

    public function getName(): string {
        return $this->name;
    }

    public function isValueSet(): bool {
        return $this->valueSet;
    }

    public function getValue() {
        return $this->value;
    }

    ####

    abstract public function getCondition(): array;

    /**
     * Set custom condition on filter
     *
     * @return static
     */
    public function setCondition(callable $conditionCallback): self {
        $this->conditionCallback = $conditionCallback;

        return $this;
    }

    public function getConditionCallback(): ?callable {
        return $this->conditionCallback;
    }

    protected function addAttributes(): BaseField {
        // if ($this->grid->hasAutoSubmit()) {
        // 	$input->setAttribute('data-autosubmit', true);
        // } else {
        // 	$input->setAttribute('data-datagrid-manualsubmit', true);
        // }

        foreach($this->attributes as $key => $value) {
            if(is_array($value)) {
                $value = array_unique($value);
                $value = implode(' ', $value);
            }

            switch($key) {
                case 'class':
                    $this->baseField->addClass($value);
                    break;
                case 'id':
                    $this->baseField->setId($value);
                    break;
                default:
                    $this->addAttribute($key, $value);
                    break;
            }
        }

        return $this->baseField;
    }

    public function render(): void {
        $params = ['field' => ['attributes' => $this->getAttributes()]];
        echo getAdminField($this->dataGrid->getFilterForm(), $this->getKey(), $params);
    }

}
