<?php

declare(strict_types=1);

namespace IZON\DataGrid\Utils;

class Paginator {

    /** @var int */
    private $base = 1;

    /** @var int */
    private $itemsPerPage;

    /** @var int */
    private $page;

    /** @var int|null */
    private $itemCount;

    // SETTER

    public function __construct(?int $itemCount = null, int $page = 1, int $itemsPerPage = 10) {
        $this->itemsPerPage = $itemsPerPage;
        $this->page = $page;
        $this->itemCount = $itemCount;
    }

    public function setPage(int $page) {
        $this->page = $page;
        return $this;
    }

    public function setItemsPerPage(int $itemsPerPage) {
        $this->itemsPerPage = max(1, $itemsPerPage);
        return $this;
    }

    public function setItemCount(int $itemCount = null) {
        $this->itemCount = $itemCount === null ? null : max(0, $itemCount);
        return $this;
    }

    // GETTER

    public function getPage(): int {
        return $this->base + $this->getPageIndex();
    }

    public function getFirstPage(): int {
        return $this->base;
    }

    public function getLastPage(): ?int {
        return $this->itemCount === null ? null : $this->base + max(0, $this->getPageCount() - 1);
    }

    protected function getPageIndex(): int {
        $index = max(0, $this->page - $this->base);
        return $this->itemCount === null
            ? $index
            : min($index, max(0, $this->getPageCount() - 1));
    }

    public function isFirst(): bool {
        return $this->getPageIndex() === 0;
    }


    public function isLast(): bool {
        return $this->itemCount === null ? false : $this->getPageIndex() >= $this->getPageCount() - 1;
    }

    public function getPageCount(): ?int {
        return $this->itemCount === null ? null : (int)ceil($this->itemCount / $this->itemsPerPage);
    }

    public function getItemsPerPage(): int {
        return $this->itemsPerPage;
    }

    public function getItemCount(): ?int {
        return $this->itemCount;
    }

    public function getOffset(): int {
        return $this->getPageIndex() * $this->itemsPerPage;
    }

    public function getLength(): int {
        if($this->itemCount === null) {
            return $this->itemsPerPage;
        }

        return min($this->itemsPerPage, $this->itemCount - $this->getPageIndex() * $this->itemsPerPage);

    }
}
