<?php

namespace IZON\DataGrid;

use IZON\Admin\MVC\ModuleInfo;
use IZON\DataGrid\DataSource\IDataSource;
use IZON\DataGrid\Exception\DataGridWrongDataSourceException;
use IZON\DataGrid\Render;
use IZON\DataGrid\Traits;
use IZON\DataGrid\Utils\Paginator;
use IZON\MVC\HttpRequest;
use IZON\MVC\Messages\HttpRequestInterface;
use Nette\Utils\Strings;

abstract class DataGrid {

    use Traits\TDataGridFilters;
    use Traits\TDataGridToolbars;
    use Traits\TDataGridActions;
    use Traits\TDataGridGroupActions;
    use Traits\TDataGridPaginator;
    use Traits\TDataGridSortAble;
    use Traits\TDataGridDetail;

    public const RENDER_TABLE_BOX_START = 'table-box-start';
    public const RENDER_TABLE_BOX_END = 'table-box-end';
    public const RENDER_TABLE = 'table';
    public const RENDER_PAGINATOR = 'paginator';
    public const RENDER_STYLE = 'style';
    public const RENDER_SCRIPT = 'script';

    public const NORENDER_AJAX = [
        self::RENDER_TABLE_BOX_START,
        self::RENDER_TABLE_BOX_END,
        //self::RENDER_STYLE,
        self::RENDER_SCRIPT,
    ];

    /** @var ModuleInfo */
    protected $module;

    /** @var array|callable[] */
    public $onRender = [];

    /** @var HttpRequest */
    protected $request;

    /** @var string */
    protected $hash;

    /** @var string */
    protected $name;

    /** @var string[] */
    protected $params;

    /** @var IDataSource|null */
    protected $datasource = null;

    /** @var Paginator */
    protected $paginator;

    /**
     * @param HttpRequest|HttpRequestInterface $request
     */
    public function __construct($request, string $name, ?IDataSource $dataSource = null) {
        $this->request = $request;
        $this->params = isset($_GET) ? $_GET : [];
        $this->params['datagrid'] = $name;

        $this->setModule($this->request->getParameter(ModuleInfo::MODULE_INFO_INDENTIFIER));
        $this->hash = md5($this->name);

        $this->paginator = new Paginator();
        $this->loadState($request);
        $this->setDatasource($dataSource);

        $this->onRender[self::RENDER_TABLE_BOX_START] = [Render\RenderTable::class, 'renderBoxStart'];
        $this->onRender[self::RENDER_TABLE] = [Render\RenderTable::class, 'render'];
        $this->onRender[self::RENDER_PAGINATOR] = [Render\RenderPaginator::class, 'render'];
        $this->onRender[self::RENDER_STYLE] = [Render\RenderStyle::class, 'render'];
        $this->onRender[self::RENDER_SCRIPT] = [Render\RenderScript::class, 'render'];
        $this->onRender[self::RENDER_TABLE_BOX_END] = [Render\RenderTable::class, 'renderBoxEnd'];
    }

    public function setModule(?ModuleInfo $module): void {
        $this->module = $module;
        if($module === null) {
            return;
        }
        $this->name = sprintf('grid_%s_%s', $this->module->getIdentifier(), $this->params['datagrid']);
    }

    public function unsetAjaxRender() {
        foreach(self::NORENDER_AJAX as $render) {
            unset($this->onRender[$render]);
        }
    }

    // SETTER

    public function setDatasource(?IDataSource $datasource): self {
        $this->datasource = $datasource;
        return $this;
    }

    public function updatePaginatorCounter(): void {
        if($this->datasource instanceof IDataSource) {
            $this->paginator->setItemCount($this->datasource->getCount());
        } else {
            $this->paginator->setItemCount(0);
        }
    }

    // GETTER

    public function getHash(): string {
        return $this->hash;
    }

    public function getName(string $part = null): string {
        if($part === null) {
            return $this->name;
        }

        return sprintf('%s_%s', $this->name, $part);
    }

    /**
     * @param string $part
     * @param mixed $default
     * @return mixed
     */
    private function getRequestParam(string $part, $default = null) {
        $param = $this->request->getParameter($this->getName($part));

        if($param === null) {
            return $default;
        }

        if(is_array($param)) {
            return $param;
        }

        if(Strings::startsWith($param, 0)) {
            return trim($param);
        }

        return ((string)intval($param) == $param) ? intval($param) : trim($param);
    }

    public function getDatasource(): ?IDataSource {
        return $this->datasource;
    }

    public function getPaginator(): Paginator {
        return $this->paginator;
    }

    // CREATOR

    public function createLink(array $_params = [], bool $resetFilter = false): string {
        $params = $this->params;

        if($resetFilter) {
            foreach(array_keys($params) as $key) {
                if(strpos($key, '_filter-') !== false) {
                    unset($params[$key]);
                }
            }
        }

        unset($params[$this->getName('groupAction')]);
        unset($params[$this->getName('groupActionIds')]);

        foreach($_params as $key => $value) {
            $params[$this->getName($key)] = $value;
        }

        $url = explode('/', $this->request->getURL());
        unset($url[count($url) - 1]);
        $url[] = 'ajaxReloadDataGrid';
        $url = implode('/', $url);

        if(empty($params)) {
            return $url;
        }

        return sprintf('%s?%s', $url, http_build_query($params));
    }

    // RENDER

    protected abstract function build(): void;

    public function render(bool $renderPopup = true) {
        if(is_null($this->datasource)) {
            throw new DataGridWrongDataSourceException(get_class($this));
        }

        $this->build();
        $this->loadFilterState();
        $this->processSortAble();

        $this->datasource->filter($this->getFilters());
        $this->processGroupAction();

        $this->updatePaginatorCounter();

        $this->datasource->limit($this->paginator->getOffset(), $this->paginator->getItemsPerPage());

        if($this->isDetail() && $this->getRequestParam('datagrid-detail')) {
            $items = $this->getDatasource()
                ->filterOne(['id' => $this->getRequestParam('datagrid-detail')])
                ->getData();

            if(isset($items[0])) {
                $render = $this->getDetail();
                $render($items[0]);
                return;
            }
        }

        foreach($this->onRender as $render) {
            $render($this);
        }

        return;

        if($renderPopup) {
            ?>
            <!-- POPUP - Confirmation of deleting item -->
            <div class="popup">
                <div class="popup__message confirmation-deleting">
                    <div class="close"></div>
                    <span><?= $this->removeConfirm ?> <strong class="zaznam"></strong>?</span>
                    <a href="#" class="btn--secondary" id="back"><?= __('Vrátit') ?></a>
                    <a href="#" class="btn--important" id="do"><?= __('Smazat') ?></a>
                </div>
            </div>
            <?php
        }
    }
}
