<?php

declare(strict_types=1);

namespace IZON\DataGrid\DataSource;

use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use Doctrine\ORM\Tools\Pagination\Paginator;
use IZON\DataGrid\AggregationFunction\IAggregatable;
use IZON\DataGrid\AggregationFunction\IAggregationFunction;
use IZON\DataGrid\Exception\DataGridDateTimeHelperException;
use IZON\DataGrid\Filter\FilterDate;
use IZON\DataGrid\Filter\FilterDateRange;
use IZON\DataGrid\Filter\FilterMultiSelect;
use IZON\DataGrid\Filter\FilterRange;
use IZON\DataGrid\Filter\FilterSelect;
use IZON\DataGrid\Filter\FilterText;
use IZON\DataGrid\Utils\DateTimeHelper;
use IZON\DataGrid\Utils\Sorting;

/**
 * @method void onDataLoaded(array $result)
 */
class DoctrineDataSource extends FilterableDataSource implements IDataSource {

    /**
     * Event called when datagrid data is loaded.
     *
     * @var array|callable[]
     */
    public $onDataLoaded;

    /** @var QueryBuilder */
    protected $dataSource;

    /** @var string */
    protected $primaryKey;

    /** @var string */
    protected $rootAlias;

    /** @var int */
    protected $placeholder;


    public function __construct(QueryBuilder $dataSource, string $primaryKey) {
        $this->placeholder = count($dataSource->getParameters());
        $this->dataSource = $dataSource;
        $this->primaryKey = $primaryKey;
    }


    public function getQuery(): Query {
        return $this->dataSource->getQuery();
    }


    /********************************************************************************
     *                          IDataSource implementation                          *
     ********************************************************************************/

    public function getCount(): int {
        if($this->usePaginator()) {
            return (new Paginator($this->getQuery()))->count();
        }

        $dataSource = clone $this->dataSource;
        $dataSource->select(sprintf('COUNT(%s) as cnt', $this->checkAliases($this->primaryKey)));
        $dataSource->resetDQLPart('orderBy');

        return (int)$dataSource->getQuery()->getSingleScalarResult();
    }


    /**
     * {@inheritDoc}
     */
    public function getData(): array {
        if($this->usePaginator()) {
            $iterator = (new Paginator($this->getQuery()))->getIterator();

            $data = iterator_to_array($iterator);
        } else {
            $data = $this->getQuery()->getResult();
        }

        //$this->onDataLoaded($data);

        return $data;
    }


    /**
     * {@inheritDoc}
     */
    public function filterOne(array $condition): IDataSource {
        $p = $this->getPlaceholder();

        foreach($condition as $column => $value) {
            $c = $this->checkAliases($column);

            $this->dataSource->andWhere("$c = :$p")
                ->setParameter($p, $value);
        }

        return $this;
    }

    /**
     * {@inheritDoc}
     */
    public function filterIn(string $column, array $inArray): IDataSource {
        $c = $this->checkAliases($column);
        $this->dataSource->addCriteria(Criteria::create()->where(Criteria::expr()->in($c, $inArray)));
        return $this;
    }


    public function limit(int $offset, int $limit): IDataSource {
        $this->dataSource->setFirstResult($offset)->setMaxResults($limit);

        return $this;
    }


    public function sort(Sorting $sorting): IDataSource {
        if(is_callable($sorting->getSortCallback())) {
            call_user_func(
                $sorting->getSortCallback(),
                $this->dataSource,
                $sorting->getSort()
            );

            return $this;
        }

        $sort = $sorting->getSort();

        if($sort !== []) {
            foreach($sort as $column => $order) {
                $this->dataSource->addOrderBy($this->checkAliases((string)$column), $order);
            }
        } else {
            /**
             * Has the statement already a order by clause?
             */
            if(!(bool)$this->dataSource->getDQLPart('orderBy')) {
                $this->dataSource->orderBy($this->checkAliases($this->primaryKey));
            }
        }

        return $this;
    }

    public function updatePosition(
        string $property,
        $current,
        int $currentPosition,
        ?int $prevPosition,
        ?int $nextPosition
    ): IDataSource {
        $table = $this->dataSource->getEntityManager()->getClassMetadata(get_class($current))->getTableName();

        $connection = $this->dataSource->getEntityManager()->getConnection();

        try {
            $connection->beginTransaction();

            if($prevPosition !== null) {
                $sql = sprintf('UPDATE %s SET %s = %s - 1 WHERE %s <= ?', $table, $property, $property, $property);
                $connection->executeQuery($sql, [$prevPosition]);
            }

            if($nextPosition !== null) {
                $sql = sprintf('UPDATE %s SET %s = %s + 1 WHERE %s >= ?', $table, $property, $property, $property);
                $connection->executeQuery($sql, [$nextPosition]);
            }

            // update current position
            $sql = sprintf('UPDATE %s SET %s = ? WHERE id = ?', $table, $property);
            $connection->executeQuery($sql, [$currentPosition, $current->getId()]);

            // update all position
            $sql = sprintf('SET @pos := 0; UPDATE %s SET %s = @pos := @pos + 1 ORDER BY %s ASC', $table, $property, $property);
            $connection->executeQuery($sql);

            $connection->commit();
        } catch(\Exception $e) {
            $connection->rollBack();
        }

        return $this;
    }

    /**
     * Get unique int value for each instance class (self)
     */
    public function getPlaceholder(): string {
        $return = 'param'.(string)($this->placeholder + 1);

        $this->placeholder++;

        return $return;
    }

    protected function applyFilterDate(FilterDate $filter): void {
        $p1 = $this->getPlaceholder();
        $p2 = $this->getPlaceholder();

        foreach($filter->getCondition() as $column => $value) {
            try {
                $date = DateTimeHelper::tryConvertToDateTime($value, [$filter->getPhpFormat()]);
                $c = $this->checkAliases($column);

                $this->dataSource->andWhere("$c >= :$p1 AND $c <= :$p2")
                    ->setParameter($p1, $date->format('Y-m-d 00:00:00'))
                    ->setParameter($p2, $date->format('Y-m-d 23:59:59'));
            } catch(DataGridDateTimeHelperException $ex) {
                // ignore the invalid filter value
            }
        }
    }


    protected function applyFilterDateRange(FilterDateRange $filter): void {
        $conditions = $filter->getCondition();
        $c = $this->checkAliases($filter->getColumn());

        $valueFrom = $conditions[$filter->getColumn()]['from'];
        $valueTo = $conditions[$filter->getColumn()]['to'];

        if($valueFrom) {
            try {
                $dateFrom = DateTimeHelper::tryConvertToDate($valueFrom, [$filter->getPhpFormat()]);
                $dateFrom->setTime(0, 0, 0);

                $p = $this->getPlaceholder();

                $this->dataSource->andWhere("$c >= :$p")->setParameter(
                    $p,
                    $dateFrom->format('Y-m-d H:i:s')
                );
            } catch(DataGridDateTimeHelperException $ex) {
                // ignore the invalid filter value
            }
        }

        if($valueTo) {
            try {
                $dateTo = DateTimeHelper::tryConvertToDate($valueTo, [$filter->getPhpFormat()]);
                $dateTo->setTime(23, 59, 59);

                $p = $this->getPlaceholder();

                $this->dataSource->andWhere("$c <= :$p")->setParameter(
                    $p,
                    $dateTo->format('Y-m-d H:i:s')
                );
            } catch(DataGridDateTimeHelperException $ex) {
                // ignore the invalid filter value
            }
        }
    }


    protected function applyFilterRange(FilterRange $filter): void {
        $conditions = $filter->getCondition();
        $c = $this->checkAliases($filter->getColumn());

        $valueFrom = $conditions[$filter->getColumn()]['from'];
        $valueTo = $conditions[$filter->getColumn()]['to'];

        if(is_numeric($valueFrom)) {
            $p = $this->getPlaceholder();
            $this->dataSource->andWhere("$c >= :$p")->setParameter($p, $valueFrom);
        }

        if(is_numeric($valueTo)) {
            $p = $this->getPlaceholder();
            $this->dataSource->andWhere("$c <= :$p")->setParameter($p, $valueTo);
        }
    }


    protected function applyFilterText(FilterText $filter): void {
        $condition = $filter->getCondition();
        $exprs = [];

        foreach($condition as $column => $value) {
            $c = $this->checkAliases($column);

            if($filter->isExactSearch()) {
                $exprs[] = $this->dataSource->expr()->eq(
                    $c,
                    $this->dataSource->expr()->literal($value)
                );

                continue;
            }

            $words = $filter->hasSplitWordsSearch() === false ? [$value] : explode(' ', (string)$value);

            foreach($words as $word) {
                $exprs[] = $this->dataSource->expr()->like(
                    $c,
                    $this->dataSource->expr()->literal("%$word%")
                );
            }
        }

        $or = call_user_func_array([$this->dataSource->expr(), 'orX'], $exprs);

        $this->dataSource->andWhere($or);
    }


    protected function applyFilterMultiSelect(FilterMultiSelect $filter): void {
        $c = $this->checkAliases($filter->getColumn());
        $p = $this->getPlaceholder();

        $values = $filter->getCondition()[$filter->getColumn()];
        $expr = $this->dataSource->expr()->in($c, ':'.$p);

        $this->dataSource->andWhere($expr)->setParameter($p, $values);
    }


    protected function applyFilterSelect(FilterSelect $filter): void {
        $p = $this->getPlaceholder();

        foreach($filter->getCondition() as $column => $value) {
            $c = $this->checkAliases($column);

            $this->dataSource->andWhere("$c = :$p")
                ->setParameter($p, $value);
        }
    }


    /**
     * {@inheritDoc}
     */
    protected function getDataSource() {
        return $this->dataSource;
    }


    private function checkAliases(string $column): string {
        if(strpos($column, '.') !== false) {
            return $column;
        }

        if(!isset($this->rootAlias)) {
            $rootAlias = $this->dataSource->getRootAliases();
            $this->rootAlias = current($rootAlias);
        }

        return $this->rootAlias.'.'.$column;
    }


    private function usePaginator(): bool {
        $hasJoin = (bool)$this->dataSource->getDQLPart('join');
        $hasGroupBy = (bool)$this->dataSource->getDQLPart('groupBy');

        return $hasJoin || $hasGroupBy;
    }

    public function __clone() {
        if(is_object($this->dataSource)) {
            $this->dataSource = clone $this->dataSource;
        }
    }
}
