<?php

namespace IZON\DataGrid\Column;

use IZON\Admin\MVC\ModuleInfo;
use IZON\DataGrid\DataGrid;
use IZON\DataGrid\Exception\DataGridColumnException;

abstract class AColumn {

    const ALIGN_LEFT = 'left';
    const ALIGN_CENTER = 'center';
    const ALIGN_RIGHT = 'right';

    const ALIGNS = [
        self::ALIGN_LEFT,
        self::ALIGN_CENTER,
        self::ALIGN_RIGHT,
    ];

    /** @var DataGrid|null */
    protected $dataGrid = null;

    /** @var ModuleInfo */
    protected $module;

    /** @var string */
    protected $key;

    /** @var string */
    protected $name;

    /** @var string */
    protected $align = self::ALIGN_LEFT;

    /** @var string */
    protected $column;

    /** @var callable|null */
    protected $render = null;

    /** @var bool */
    protected $sortable = false;

    public function __construct(string $key, string $name, ?string $column = null) {
        $this->key = $key;
        $this->name = $name;
        $this->column = $column ? $column : $key;
    }

    // GETTER

    public function getDataGrid(): DataGrid {
        return $this->dataGrid;
    }

    public function getName(): string {
        return $this->name;
    }

    public function getValue($item): string {
        return $this->processRenderFunction($item);
    }

    // SETTER

    public function setDataGrid(DataGrid $dataGrid): void {
        if(!$this->dataGrid instanceof DataGrid) {
            $this->dataGrid = $dataGrid;
        }
    }

    public function setModule(ModuleInfo $module): void {
        $this->module = $module;
    }

    public function setRender(?callable $render): self {
        $this->render = $render;
        return $this;
    }

    public function setAlign(string $align): self {
        if(in_array($align, self::ALIGNS)) {
            $this->align = $align;
        } else {
            $this->align = self::ALIGN_LEFT;
        }
        return $this;
    }

    public function setSortable(bool $sortable = true): self {
        $this->sortable = $sortable;
        return $this;
    }

    // RENDER

    public function renderTh(): void {
        $sortable = null;
        $class = [];

        if($this->sortable) {
            //$sortable = '<span class="sortable__up"></span><span class="sortable__down"></span>';
            $sortable = 'SORTING';
            $class[] = 'sortable';
        }

        echo sprintf(
            '<th class="datagrid-align-%s %s">%s%s</th>',
            $this->align,
            implode(' ', $class),
            $this->name,
            $sortable
        );
    }

    public function renderTd($item): void {
        echo sprintf('<td class="datagrid-align-%s">%s</td>', $this->align, $this->processRenderFunction($item));
    }

    protected function processRenderFunction($item): ?string {
        if(!is_null($this->render)) {
            $callable = $this->render;
            return $callable($item);
        }

        return $this->render($item);
    }

    protected function render($item) {
        return $this->getProperty($item, $this->column, true);
    }

    protected function getProperty($item, string $column, bool $throw = false) {
        $property = ucfirst($column);

        $methodGet = sprintf('get%s', $property);
        $methodIs = sprintf('is%s', $property);

        if(is_array($item) && isset($item[$column])) {
            $value = $item[$column];
        } elseif(is_object($item) && method_exists($item, $methodGet)) {
            $value = $item->$methodGet();
        } elseif(is_object($item) && method_exists($item, $methodIs)) {
            $value = $item->$methodIs();
        } elseif($throw) {
            throw new DataGridColumnException($item, $column);
        } else {
            return $column;
        }

        return $value;
    }

}
