<?php

declare (strict_types=1);
namespace IZON\Application\Tools\Rector;

use PhpParser\Node;
use PhpParser\Node\Stmt\Class_;
use Rector\Rector\AbstractRector;
use ReflectionClass;
use Symplify\RuleDocGenerator\ValueObject\CodeSample\CodeSample;
use Symplify\RuleDocGenerator\ValueObject\RuleDefinition;
use \PhpParser\Node\Stmt\Property;

final class RemoveOverridingPropertiesRector extends AbstractRector
{

    public function getRuleDefinition(): RuleDefinition
    {
        return new RuleDefinition(
            'Removes public and protected properties that override parent class properties',
            [new CodeSample(<<<'CODE_SAMPLE'
class SomeParent
{
    public string $name;
}
class SomeClass extends SomeParent
{
    public $name;
}
CODE_SAMPLE
            , <<<'CODE_SAMPLE'
class SomeParent
{
    public string $name;
}
class SomeClass extends SomeParent
{
}
CODE_SAMPLE
        )]);
    }
    /**
     * @return array<class-string<Node>>
     */
    public function getNodeTypes(): array
    {
        return [Class_::class];
    }
    /**
     * @param Class_ $node
     */
    public function refactor(Node $node): ?Node
    {
        if($node->extends === null) {
            return null;
        }

        $parentClassReflection = new ReflectionClass($this->getName($node->extends));
        $parentProperties = $this->getParentClassPublicProtectedPropertiesNames($parentClassReflection);

        $hasChanged = false;
        foreach ($node->stmts as $index => $statement) {
            if (!$statement instanceof Property) {
                continue;
            }
            if (array_key_exists($this->getName($statement), $parentProperties)) {
                unset($node->stmts[$index]);
                $hasChanged = true;
            }
        }
        if ($hasChanged) {
            return $node;
        }
        return null;
    }

    private function getParentClassPublicProtectedPropertiesNames(\ReflectionClass $reflectionClass): array
    {
        $propertiesHashMap = [];
        foreach ($reflectionClass->getProperties() as $property) {
            if (!$property->isPublic() && !$property->isProtected()) {
                continue;
            }
            $propertiesHashMap[$property->getName()] = $property->getName();
        }

        if ($reflectionClass->getParentClass()) {
            $propertiesHashMap = array_merge(
                $propertiesHashMap,
                $this->getParentClassPublicProtectedPropertiesNames($reflectionClass->getParentClass())
            );
        }

        return $propertiesHashMap;
    }
}
