<?php

declare(strict_types=1);

namespace IZON\Application\Tools\Rector\DB;

use PhpParser\Node;
use PhpParser\Node\Name;
use PhpParser\Node\Stmt\Class_;
use PhpParser\NodeAbstract;
use PHPStan\PhpDocParser\Ast\Type\IdentifierTypeNode;
use PHPStan\PhpDocParser\Ast\Type\TypeNode;
use Rector\BetterPhpDocParser\PhpDocInfo\PhpDocInfoFactory;
use Rector\BetterPhpDocParser\ValueObject\Type\BracketsAwareUnionTypeNode;
use Rector\Rector\AbstractRector;
use Symplify\RuleDocGenerator\ValueObject\CodeSample\CodeSample;
use Symplify\RuleDocGenerator\ValueObject\RuleDefinition;

final class EntityAddStrictTypeFromAnnotationRector extends AbstractRector
{

    const ENTITY_ATTRIBUTE_CLASS = 'Doctrine\ORM\Mapping\Entity';
    /**
     * @readonly
     */
    private PhpDocInfoFactory $phpDocInfoFactory;

    public function __construct(PhpDocInfoFactory $phpDocInfoFactory)
    {
        $this->phpDocInfoFactory = $phpDocInfoFactory;
    }

    public function getRuleDefinition(): RuleDefinition
    {
        return new RuleDefinition('Add T to QueryResult<T> annotation in Dao classes', [new CodeSample(
            <<<'CODE_SAMPLE'
use Doctrine\ORM\Mapping as ORM;

#[ORM\Entity]
interface EntityDao extends Dao
{
    /**
     * @var null|int
     */
    #[ORM\Column]
    #[ORM\GeneratedValue]
    #[ORM\Id]
    protected $id;

    /**
     * @var string|null
     */
    #[ORM\Column(nullable: true)]
    protected $externalId;

    /**
     * @var bool
     */
    #[ORM\Column]
    protected $active = false;
}
CODE_SAMPLE
            ,
            <<<'CODE_SAMPLE'
use Doctrine\ORM\Mapping as ORM;

#[ORM\Entity]
interface EntityDao extends Dao
{
    /**
     * @var null|int
     */
    #[ORM\Column]
    #[ORM\GeneratedValue]
    #[ORM\Id]
    protected ?int $id = null;

    /**
     * @var string|null
     */
    #[ORM\Column(nullable: true)]
    protected ?string $externalId;

    /**
     * @var bool
     */
    #[ORM\Column]
    protected bool $active = false;
}
CODE_SAMPLE
        )]);
    }

    public function getNodeTypes(): array
    {
        return [Class_::class];
    }

    /**
     * @param Class_ $node
     */
    public function refactor(Node $node): ?Node
    {
        if (!$this->isEntity($node)) {
            return null;
        }
        $this->addTypesToProperties($node);
        return $node;
    }

    protected function isEntity(Class_ $class): bool
    {
        foreach ($class->attrGroups as $attrGroup) {
            foreach ($attrGroup->attrs as $attr) {
                if ($this->isName($attr->name, self::ENTITY_ATTRIBUTE_CLASS)) {
                    return true;
                }
            }
        }
        foreach ($class->extends as $extend) {
            $extendClass = new Class_($extend);
            $isEntity = $this->isEntity($extendClass);
            if($isEntity) {
                return true;
            }
        }
        return false;
    }

    private function addTypesToProperties(Class_ $class): void
    {
        foreach ($class->getProperties() as $property) {
            if ($property->type !== null) { // has type
                // skip already typed properties
                continue;
            }

            $phpDocInfo = $this->phpDocInfoFactory->createFromNodeOrEmpty($property);

            $varTag = $phpDocInfo->getVarTagValueNode();
            if ($varTag === null) { // no @var tag
                continue;
            }

            $typeNode = $this->getTypeNode($varTag->type);
            if ($typeNode) {
                $property->type = $typeNode;
            }

            // set default null for nullable properties without default value
            foreach ($property->props as $prop) {
                if (
                    $prop->default === null
                    && $typeNode instanceof Node\NullableType
                ) {
                    $prop->default = new Node\Expr\ConstFetch(new Name('null'));
                }
            }
        }
    }

    protected function getTypeNode(TypeNode $typeNode): ?NodeAbstract
    {
        $nullable = false;
        $mainType = null;
        if ($typeNode instanceof BracketsAwareUnionTypeNode) {
            $types = $typeNode->types;
            if (count($typeNode->types) > 2) { // too complex type
                return null;
            }

            foreach ($types as $type) {
                if ($type->name == 'null') {
                    $nullable = true;
                } elseif($mainType === null) {
                    $mainType = $type->name;
                } else {
                    // more than one non-null type
                    return null;
                }
            }
        } elseif ($typeNode instanceof IdentifierTypeNode) {
            $mainType = $typeNode->name;
        }
        if($mainType === null) {
            return null;
        }

        if ($mainType == 'integer') {
            $mainType = 'int';
        } elseif ($mainType == 'boolean') {
            $mainType = 'bool';
        } elseif ($mainType == 'double') {
            $mainType = 'float';
        }

        $newTypeName = new Name($mainType);
        if ($nullable) {
            $newTypeName = new Node\NullableType($newTypeName);
        }
        return $newTypeName;
    }
}
