<?php

declare(strict_types=1);

namespace IZON\Application\Tools\Rector\DB;

use PhpParser\Node;
use PhpParser\Node\Expr\ClassConstFetch;
use PhpParser\Node\Stmt\Interface_;
use PHPStan\PhpDocParser\Ast\PhpDoc\GenericTagValueNode;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocTagNode;
use Rector\BetterPhpDocParser\PhpDocInfo\PhpDocInfoFactory;
use Rector\Comments\NodeDocBlock\DocBlockUpdater;
use Rector\Rector\AbstractRector;
use Symplify\RuleDocGenerator\ValueObject\CodeSample\CodeSample;
use Symplify\RuleDocGenerator\ValueObject\RuleDefinition;

final class AddAnnotationToDaoRector extends AbstractRector
{
    protected const DAO_CLASS = 'IZON\DB\Dao';

    /**
     * @readonly
     */
    private DocBlockUpdater $docBlockUpdater;

    /**
     * @readonly
     */
    private PhpDocInfoFactory $phpDocInfoFactory;

    public function __construct(DocBlockUpdater $docBlockUpdater, PhpDocInfoFactory $phpDocInfoFactory)
    {
        $this->docBlockUpdater = $docBlockUpdater;
        $this->phpDocInfoFactory = $phpDocInfoFactory;
    }

    public function getRuleDefinition(): RuleDefinition
    {
        return new RuleDefinition('Add @extends Dao<T> annotation to Dao classes', [new CodeSample(
            <<<'CODE_SAMPLE'
namespace Dao;

use Entity;
use IZON\DB\Dao;

interface EntityDao extends Dao
{
    public const DOMAIN_CLASS = Entity::class;
}
CODE_SAMPLE
            ,
            <<<'CODE_SAMPLE'
namespace Dao;

use Entity;
use IZON\DB\Dao;

/**
 * @extends Dao<Entity>
 */
interface EntityDao extends Dao
{
    public const DOMAIN_CLASS = Entity::class;
}
CODE_SAMPLE
        )]);
    }
    /**
     * @return array<class-string<Node>>
     */
    public function getNodeTypes(): array
    {
        return [Interface_::class];
    }

    /**
     * @param Interface_ $node
     */
    public function refactor(Node $node): ?Node
    {
        if (!$this->isDaoClass($node)) {
            return null;
        }

        $domainClass = $this->getDomainClassFromConstant($node);
        if ($domainClass === null || $this->hasExtendsAnnotation($node)) {
            return null;
        }
        $this->addAnnotationToNode($node, $domainClass);
        return $node;
    }

    protected function isDaoClass(Interface_ $class): bool
    {
        foreach ($class->extends as $extend) {
            return $this->isName($extend, self::DAO_CLASS);
        }
        return false;
    }

    protected function getDomainClassFromConstant(Interface_ $class): ?string
    {
        $classConstants = $class->getConstants();
        foreach ($classConstants as $classConst) {
            foreach ($classConst->consts as $constNode) {
                if (
                    $constNode->name == 'DOMAIN_CLASS'
                    && $constNode->value instanceof ClassConstFetch
                ) {
                    return $constNode->value->class->name;
                }
            }
        }
        return null;
    }

    private function addAnnotationToNode(Interface_ $class, string $domainClass): void
    {
        $phpDocInfo = $this->phpDocInfoFactory->createFromNodeOrEmpty($class);
        $genericsAnnotation = \sprintf('\\%s<\\%s>', self::DAO_CLASS, $domainClass);
        $phpDocInfo->addPhpDocTagNode(new PhpDocTagNode('@extends', new GenericTagValueNode($genericsAnnotation)));
        $this->docBlockUpdater->updateRefactoredNodeWithPhpDocInfo($class);
    }

    private function hasExtendsAnnotation(Interface_ $class): bool
    {
        return $this->phpDocInfoFactory->createFromNodeOrEmpty($class)->hasByName('@extends');
    }
}
