<?php

declare(strict_types=1);

namespace IZON\Application\Tools\Rector\DB;

use PhpParser\Node;
use PhpParser\Node\Expr\ClassConstFetch;
use PhpParser\Node\Stmt\ClassMethod;
use PhpParser\Node\Stmt\Interface_;
use PHPStan\PhpDocParser\Ast\PhpDoc\ReturnTagValueNode;
use PHPStan\PhpDocParser\Ast\Type\GenericTypeNode;
use PHPStan\PhpDocParser\Ast\Type\IdentifierTypeNode;
use Rector\BetterPhpDocParser\PhpDocInfo\PhpDocInfo;
use Rector\BetterPhpDocParser\PhpDocInfo\PhpDocInfoFactory;
use Rector\Comments\NodeDocBlock\DocBlockUpdater;
use Rector\Rector\AbstractRector;
use Symplify\RuleDocGenerator\ValueObject\CodeSample\CodeSample;
use Symplify\RuleDocGenerator\ValueObject\RuleDefinition;

final class AddAnnotationQueryResultRector extends AbstractRector
{
    protected const DAO_CLASS = 'IZON\DB\Dao';

    protected const QUERY_RESULT_CLASS = 'IZON\DB\QueryResult';

    /**
     * @readonly
     */
    private DocBlockUpdater $docBlockUpdater;

    /**
     * @readonly
     */
    private PhpDocInfoFactory $phpDocInfoFactory;

    public function __construct(DocBlockUpdater $docBlockUpdater, PhpDocInfoFactory $phpDocInfoFactory)
    {
        $this->docBlockUpdater = $docBlockUpdater;
        $this->phpDocInfoFactory = $phpDocInfoFactory;
    }

    public function getRuleDefinition(): RuleDefinition
    {
        return new RuleDefinition('Add T to QueryResult<T> annotation in Dao classes', [new CodeSample(
            <<<'CODE_SAMPLE'
namespace Dao;

use Entity;
use IZON\DB\Dao;

interface EntityDao extends Dao
{
    public const DOMAIN_CLASS = Entity::class;

    public function findAll(): QueryResult;
}
CODE_SAMPLE
            ,
            <<<'CODE_SAMPLE'
namespace Dao;

use Entity;
use IZON\DB\Dao;

/**
 * @extends Dao<Entity>
 */
interface EntityDao extends Dao
{
    public const DOMAIN_CLASS = Entity::class;

    /**
     * @return QueryResult<Entity>
     */
    public function findAll(): QueryResult;
}
CODE_SAMPLE
        )]);
    }

    public function getNodeTypes(): array
    {
        return [Interface_::class];
    }

    /**
     * @param Interface_ $node
     */
    public function refactor(Node $node): ?Node
    {
        if (!$this->isDaoClass($node)) {
            return null;
        }
        $domainClass = $this->getDomainClassFromConstant($node);
        if ($domainClass === null) {
            return null;
        }
        $this->addAnnotationsToQueryResult($node, $domainClass);
        return $node;
    }

    protected function isDaoClass(Interface_ $class): bool
    {
        foreach ($class->extends as $extend) {
            return $this->isName($extend, self::DAO_CLASS);
        }
        return false;
    }

    protected function getDomainClassFromConstant(Interface_ $class): ?string
    {
        $classConstants = $class->getConstants();
        foreach ($classConstants as $classConst) {
            foreach ($classConst->consts as $constNode) {
                if (
                    $constNode->name == 'DOMAIN_CLASS'
                    && $constNode->value instanceof ClassConstFetch
                ) {
                    return $constNode->value->class->name;
                }
            }
        }
        return null;
    }

    private function addAnnotationsToQueryResult(Interface_ $class, string $domainClass): void
    {
        foreach ($class->getMethods() as $method) {
            $phpDocInfo = $this->phpDocInfoFactory->createFromNodeOrEmpty($method);

            if (!$this->isReturnQueryFactory($method, $phpDocInfo)) {
                continue;
            }

            $tagValueNode = new GenericTypeNode(
                new IdentifierTypeNode(self::QUERY_RESULT_CLASS),
                [new IdentifierTypeNode($domainClass)]
            );

            $returnTagValue = $phpDocInfo->getReturnTagValue();
            if ($returnTagValue === null) { // add new tag
                $returnTagNode = new ReturnTagValueNode($tagValueNode, '');
                $phpDocInfo->addTagValueNode($returnTagNode);
            } else {
                $returnTagValue->type = $tagValueNode;
            }

            $this->docBlockUpdater->updateRefactoredNodeWithPhpDocInfo($method);
        }
    }

    private function isReturnQueryFactory(ClassMethod $method, PhpDocInfo $phpDocInfo): bool
    {
        if (
            $method->getReturnType()
            && $this->isName($method->getReturnType(), self::QUERY_RESULT_CLASS)
        ) {
            return true;
        }
        $returnTagValue = $phpDocInfo->getReturnTagValue();
        if (!$returnTagValue instanceof ReturnTagValueNode) {
            return false;
        }
        $returnTypeName = $returnTagValue->type->name;
        if (
            $returnTypeName == self::QUERY_RESULT_CLASS // TODO: get fully qualified name
            || $returnTypeName == 'QueryResult'
        ) {
            return true;
        }
        return false;
    }
}
