<?php
namespace IZON\Admin\Services;

use IZON\DB\QueryParams\OrderBy;
use IZON\DB\Repository\RepositoryInterface;
use Exception;

use IZON\DB\EntityManager\EntityManager;
use IZON\Utils\Slug;
use function IZON\Object\createGetterMethodName;
use function IZON\Object\createIsMethodName;
use function IZON\Object\createSetterMethodName;

/**
 * Base servis pro administraci - obsahuje metody vyuzivane AbstractBaseControllerem
 * Obsahuje metody pro changeSimpleProperty
 * dale metody pro praci s obrazky standartaně v db vazba 1:N
 */
abstract class AbstractBaseService {

    /**
     * @var EntityManager
     */
    protected $entityManager;

    /**
     * @var RepositoryInterface
     */
    protected $entityRepository;


    /**
     * @param EntityManager $entityManager entity managet to access to access db entities
     * @param string $entityClassName
     */
    public function __construct(EntityManager $entityManager, string $entityClassName) {
        $this->entityManager = $entityManager;
        $this->entityRepository = $entityManager->getRepository($entityClassName);
    }


    /**
     * toggles boolean value of $propertyName of entity with id $entityId
     * @param mixed $entityId - id of updated object
     * @param string $propertyName - name of boolean property (atribute), 
     */
    public function toggleEntityProperty($entityId, string $propertyName) {
        $object = $this->entityRepository->find($entityId);
        if($object != false) {
            $getterGet = createGetterMethodName($propertyName);
            $getterIs = createIsMethodName($propertyName);
            $setter = createSetterMethodName($propertyName);
            $method = null;

            if (method_exists($object, $getterGet)){
                $method = $getterGet;
            } else if(method_exists($object, $getterIs)){
                $method = $getterIs;
            } else {
                throw new \Exception('Class: '. get_class($object) . ' does not have method ' . $getterIs . ' or ' . $getterGet);
            }

            if( method_exists($object, $setter) ) {
                $object->$setter(!$object->$method());
                $this->entityManager->flush();
            } else {
                throw new \Exception('Class: '. get_class($object) . ' does not have method  ' . $setter);
            }
        }
    }

    /**
     * changes sequence number of entity, reindexes all other entities in same squence
     * @param mixed $entityId id of entity to change sequence number
     * @param int $sequenceNumber new sequence number of passed entity
     * @param string $sequenceNumberPropertyName property used to order sequence
     * @param string $orderType if to index sequence asc or desc
     * @param array $sequenceSelector selector to select elements from sequence
     */
    public function changeEntitySequenceNumber(
        $entityId,
        int $sequenceNumber,
        string $sequenceNumberPropertyName,
        $orderType = OrderBy::ORDER_ASC,
        array $sequenceSelector = []
    ) {
        $setter = createSetterMethodName($sequenceNumberPropertyName);

        $elements = $this->entityRepository->findBy($sequenceSelector, [$sequenceNumberPropertyName => $orderType]);

        // reindex sequence number to be unique for every entity and start from
        foreach($elements as $index => $element) {
            $element->$setter($index);
        }

        $obj = $this->entityRepository->find($entityId);
        // set sequence number for required entity
        $obj->$setter($sequenceNumber);


        $newSeqNum = 0;
        if( $sequenceNumber == 0 ) { // shift element to first position
            $newSeqNum = 1;
        }
        // reindex other elements
        foreach($elements as $element) {
            if($element->getId() == $obj->getId()) {
                // do not modify changed object
                continue;
            } else if( $newSeqNum == $sequenceNumber ) {
                $newSeqNum++; // skip sequence number for changes
            } else {
                $element->$setter($newSeqNum++);
            }
        }

        $this->entityManager->flush();
    }
    
    
    /********************** IMAGE methods ****************************************/
    /*****************************************************************************/
    
    /**
     * Otoci obrazek
     * Turn image which DBobject contain 
     * @param DBObject $image - object, ktery obsahuje obrazek ( DB object musi obsahovat atribut turn a metodu getTurn())
     * @param string $method - nazev metody ktere vrati obrazek z DBObjektu
     */
    public function turnImage($image, $method = 'getImage'){
        if($image->getTurn() != 0 && $image->getTurn() != ''){
            $picture = $image->$method();
            $path = ($picture->getFsPath());

            //nacteni
            $mine = $picture->getMineType();
            switch ($mine) {
                case "image/gif":
                    $img = imagecreatefromgif($path);
                    break;
                case "image/jpeg":
                    $img = imagecreatefromjpeg($path);
                    break;
                case "image/png":
                    $img = imagecreatefrompng($path);
                    imagealphablending($img, false);
                    imagesavealpha($img, true);
                    break;
                default:
                    throw new Exception('Unknown image type.');
            }
            //rotace
            $rotate = imagerotate($img, $image->getTurn(), 0); 

            //ulozeni
            switch ($mine) {
                case "image/gif":
                    imagegif($rotate, $path);
                    break;
                case "image/jpeg":
                    imagejpeg($rotate, $path);
                    break;
                case "image/png":
                    imagealphablending( $rotate, false);
                    imagesavealpha($rotate, true);
                    imagepng($rotate, $path);
                    break;
            }

            imagedestroy($img);
            imagedestroy($rotate);
        }
        
    }
    
    /**
     * Metoda pro ukladani std. obrazku - save/update
     * !!  Doporučuji tuto metodu obalit transakci !!!
     * @param DBObject $obj - $object which contain images
     * @param string $methodName - name of method to get images from $obj
     * @param string $connPropName property name that realises connection to $obj
     */
    public function updateImages($obj, $methodName = 'getImages', $connPropName = "fkObjectId") {
        $this->checkSetImageDao();
        // zpracovani obrazku
        $images = $this->imageDao->find([$connPropName => $obj->getId()])->listResult();
        $imagesHash = [];
        foreach($images as $image) {
            $imagesHash[$image->getId()] = $image;
        }
        
        foreach($obj->$methodName() as $newImage) {
            //otoceni
            $turn = 0;
            if($newImage->getTurn() != null && $newImage->getTurn() != 0 ){
                $turn = $newImage->getTurn();
            }
            
            $connectionSetterName = "set". ucfirst($connPropName);
            $newImage->$connectionSetterName($obj->getId());
            
            if(isset($imagesHash[$newImage->getId()]) ) {
                $oldImage = $imagesHash[$newImage->getId()];
                $newImage->setImage($oldImage->getImage());
                $this->imageDao->update($newImage);
                unset($imagesHash[$newImage->getId()]);
            } else {
                $this->imageDao->save($newImage);
            }
            
            if($turn != 0){
                $newImage->setTurn($turn);
                $this->turnImage($newImage);
            }
        }
        // odstranit smazane
        foreach($imagesHash as $image) {
            $this->imageDao->delete($image);
        }
    }
    
    /**
     * Smaže všechny obrazky k danému objektu
     *  * !!  Doporučuji tuto metodu obalit transakci !!!
     * @param object $objectId - id objektu ke smazani
     * @param string $connPropName property name that realises connection to $obj
     */
    public function deleteAllImages($objectId, $connPropName = "fkObjectId") {
        $this->checkSetImageDao();
        $images = $this->imageDao->find([$connPropName => $objectId])->listResult();
        foreach ($images as $image) {
            $this->imageDao->delete($image);
        }
    }
   
    /**
     * Return images by fk, in order by column position
     * Method try to order by position on failld return order by default
     * @param integer $fkObjId ID of FK key
     * @param string $tableName Name of images table, Have not be set TODO table name isn't needed any more, mabe remove in future?
     * @return array Array of images
     */
    public function getImagesByFkId($fkObjId, $tableName = NULL, $connPropName = "fkObjectId"){
        $this->checkSetImageDao();
        try{
            $images = $this->imageDao->find([$connPropName => $fkObjId], ['position' => OrderBy::ASC])->listResult();
        }catch (Exception $e){
            $images = $this->imageDao->find([$connPropName => $fkObjId])->listResult();
        }

        return $images;
    }
    
    
    /************* PROTECTED and PRIVATE method *******************************/

    /**
     * Make unique slug in table
     * @param string  $name string from which I make slug
     * @param Dao $dao Dao class of table
     * @param string $columnName column where is save slug string
     * @return string Slug
     */
    protected function createUniqueSlugInTable($name, Dao $dao, $columnName = 'slug'){
        $slug = Slug::createSlug($name);
        $res = $dao->find([$columnName => $slug])->listResult();
        if(!empty($res)){
            $num = 0;
            do{
                $num++;
                $res = $dao->find([$columnName => $slug.'-'.$num])->listResult();
            }while(!empty($res));
            $slug = $slug . '-' . $num;
        }

        return $slug;
    }

    /**
     * Check if is set atribute dao
     * @throws Exception
     */
    protected function checkSetDao(){
        if(empty($this->dao)){
            throw new \Exception('Atribute dao is not set');
        }
    }
    
    /**
     * Check if is set atribute imageDao
     * @throws Exception
     */
    protected function checkSetImageDao(){
        if(empty($this->imageDao)){
            throw new \Exception('Atribute imageDao is not set');
        }
    }
    
    
    
}
