<?php

namespace IZON\Admin\Services\Impl;

use Doctrine\Common\Persistence\ObjectRepository;
use Exception;
use IZON\Admin\Config;
use IZON\Admin\Domain\AdminModule;
use IZON\Admin\Domain\AdminUser;
use IZON\Admin\Presentation\Domain\AdminModuleActionInfo;
use IZON\Admin\Presentation\Domain\AdminModuleInfo;
use IZON\Admin\Repositories\AdminModuleRepository;
use IZON\Admin\Services\AdminServiceInterface;
use IZON\Admin\Services\LoggedAdminUserSessionServiceInterface;
use IZON\DB\EntityManagerInterface;
use IZON\DB\Repository\BaseRepository;
use IZON\DBLocale\Domain\DBCountry;
use IZON\DBLocale\Domain\DBLocale;
use IZON\MVC\Gettext\GettextLocaleManager;
use IZON\Utils\Locale;
use function IZON\Arrays\isEmpty;

/**
 * Servis provadejici spravu uzivatelske session ziskavane z db AdminUser
 */
class LoggedAdminUserSessionService implements LoggedAdminUserSessionServiceInterface {

    protected AdminServiceInterface $adminService;

    /**
     * @var string
     */
    protected string $configDir;

    /**
     * @var array inforamtions about actions of all modules
     */
    protected array $modulesActions;

    /**
     * @var array directories od all installed modules
     */
    protected array $modulesDirs;

    protected AdminModuleRepository $adminModuleRepsitory;

    protected BaseRepository $adminUserRepsitory;

    protected BaseRepository $localeRepsitory;

    protected BaseRepository $coutryRepsitory;

    /**
     * @var string pod jakym jmenem je uzivatel ulozen v session
     */
    protected string $adminLoggedUserSessionName = "adminLoggedAdminUser";

    /**
     * @var string how is identifierd selected country in session
     */
    protected string $adminSelectedCountrySessionName = "adminSelectedCountry";

    /**
     * @var string how is identifierd selected locale of admin interface in session
     */
    protected string $adminSelectedInterfaceLocaleSessionName = "adminSelectedInterfaceLocale";

    /**
     * @var GettextLocaleManager
     */
    protected GettextLocaleManager $gettextLocaleManager;

    /**
     * @var array cache for actions of modules
     */
    protected array $modulesTypesActionsCache = [];

    /**
     * @var array  contains [roleId => [countryId1, countryId2]]
     */
    protected array $roleSupportedCountries = [];

    /**
     * @var null|string hash of universal password that can be used for login of all users
     */
    protected ?string $universalPasswordHash = null;


    function __construct(
        EntityManagerInterface $entityManager,
        AdminServiceInterface $adminService,
        $configDir,
        array $modulesActions,
        array $modulesDirs,
        GettextLocaleManager $gettextLocaleManager
    ) {
        $this->adminService = $adminService;
        $this->configDir = $configDir;
        $this->modulesActions = $modulesActions;
        $this->modulesDirs = $modulesDirs;
        $this->gettextLocaleManager = $gettextLocaleManager;

        $this->adminModuleRepsitory = $entityManager->getRepository(AdminModule::class);
        $this->adminUserRepsitory = $entityManager->getRepository(AdminUser::class);
        $this->localeRepsitory = $entityManager->getRepository(DBLocale::class);
        $this->coutryRepsitory = $entityManager->getRepository(DBCountry::class);
    }


    /**
     * prihlasi uzivatele nebo vyhodi vyjimku proc to neni mozne
     * @param string $login
     * @param string $password
     */
    public function loginUser($login, $password) {
        // TODO nepouzivat global
        /* @var $user AdminUser */
        $user = $this->adminUserRepsitory->findOneBy(["login" => $login]);
        if( $user == NULL) {
            throw new Exception("Pro login $login neexistuje uzivatel", 100);
        }
        if(
            !$this->isPasswordCorrect($user, $password)
            && !password_verify($password, $this->universalPasswordHash)
        ) {
            throw new Exception("Pro login $login nesouhlasí heslo", 200);
        }
        if( $user->getActive() == 0 ) {
            throw new Exception("Uzivatel $login neni aktivni", 300);
        }

//        $country = $this->countryDao->load($user->getFkCountryId());
////        var_dump($country);
//        if( $country == NULL ) {
//            throw new Exception("Login $login nemá nastavenou defaultni zem", 0);
//        }

        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->adminLoggedUserSessionName]      = $user->getId();
        $_SESSION[$this->adminSelectedCountrySessionName] = $user->getCountry();
        /* @var $locale \IZON\DBLocale */
        $_SESSION[$this->adminSelectedInterfaceLocaleSessionName] = $user->getInterfaceLocale()->getId();
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    /**
     * odhlasi v soucasnosti prihlaseneho uzivatele
     */
    public function logoutUser() {
        // nastartuj session
        session_start();
        // a zlikviduj session
        session_destroy();
    }

    /**
     * zkontroluje jestli zadane heslo odpovida zadanemu
     * @param AdminUser $user
     */
    protected function isPasswordCorrect($user, $password) {
        return password_verify($password , $user->getPassword());
    }

    public function isUserLoggedIn() {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }

        $retval = isset($_SESSION[$this->adminLoggedUserSessionName]);

        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    public function canBeUserLoggedIn() {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }

        if( !isset($_SESSION[$this->adminLoggedUserSessionName]) ) {
            throw new Exception("User isn't logged in");
        }
        /** @var int $loggedUserId */
        $loggedUserId = $_SESSION[$this->adminLoggedUserSessionName];

        /* @var $userCurrentState AdminUser */
        $userCurrentState = $this->adminUserRepsitory->find($loggedUserId);
        if( $userCurrentState === null ) {
            throw new Exception("Logged user with id = ". $loggedUserId ." does not exist.");
        }

        $retval = $userCurrentState->getActive();

        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    public function getLoggedUser() {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        if( !isset($_SESSION[$this->adminLoggedUserSessionName]) ) {
            throw new Exception("User isn't logged in");
        }
        $retval = $_SESSION[$this->adminLoggedUserSessionName];
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        $adminUser = $this->adminUserRepsitory->find($retval);
        if( $adminUser === null ) {
            throw new Exception("Logged user with id = ". $retval ." does not exist.");
        }

        return $adminUser;
    }

    function getLoggedUserModules() {
        if( !$this->isUserLoggedIn() ) {
            return [];
        }

        $loggedUser = $this->getLoggedUser();
        $interfaceLocale = $this->getSelectedInterfaceLocale();
        $loggedUserModules = [];
        if( $loggedUser->getSuperuser() ) {
            // TODO: rozlisit jestli je uzivatel superuser nebo ne
            $loggedUserModules = $this->adminModuleRepsitory->findAdminVisibleModules()->getResult();
        } else {
            $loggedUserModules = $this->adminModuleRepsitory->findUserAdminVisibleModules($loggedUser->getId())->getResult();
        }

        $loggedUserModulesInfo = [];
        foreach($loggedUserModules as $adminModule) {
            $loggedUserModulesInfo[] = $this->createAdminModuleInfo($adminModule, $interfaceLocale->getId());
        }

        return $loggedUserModulesInfo;
    }

    public function getUserCountries() {
        $user = $this->getLoggedUser();
        $userCountryIdes = [];
        foreach($user->getRoles() as $role) {
            if( array_key_exists($role->getId(), $this->roleSupportedCountries) ) {
                $userCountryIdes = array_merge($userCountryIdes, $this->roleSupportedCountries[$role->getId()]);
            }
        }

        $countries = [];
        if( isEmpty($userCountryIdes) ) {
            $countries = $this->coutryRepsitory->findAll();
        } else {
            foreach($userCountryIdes as $userCountryId) {
                $countries[] = $this->coutryRepsitory->find($userCountryId);
            }
        }
        return $countries;
    }

    public function getSelectedCountry() {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        if( !isset($_SESSION[$this->adminSelectedCountrySessionName]) ) {
            throw new Exception("Selected country isn't set");
        }
        $retval = $_SESSION[$this->adminSelectedCountrySessionName];
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    /**
     * nastavuje prave vybranou zemi
     */
    public function setSelectedCountry($countryCode) {
        $country = $this->coutryRepsitory->findOneBy(["code" => $countryCode]);
        if( $country == NULL ) {
            throw new Exception("Country with code $countryCode not supported");
        }

        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->adminSelectedCountrySessionName] = $country;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    public function getSupportedInterfaceLocales() {
        return $this->adminService->getSupportedInterfaceLocales();
    }

    public function getSelectedInterfaceLocale() {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }

        if( !isset($_SESSION[$this->adminSelectedInterfaceLocaleSessionName]) ) {
            throw new Exception("Selected Interface locale isn't set"); // DOTO: po nejake dobe odkometovat, je to zakomentovane kvuli tomu, aby se nerozbilo prihlaseny, po nejake bobe uz busou vsichni prihlaseni se spravnym locale
        } else {
            $localeId = $_SESSION[$this->adminSelectedInterfaceLocaleSessionName];
            $locale = $this->localeRepsitory->find($localeId);
            $retval = $locale;
        }

        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }

        return $retval;
    }

    public function setSelectedInterfaceLocale($localeTag) {
        /** @var DBLocale $locale */
        $locale = $this->localeRepsitory->findOneBy(["localeTag" => $localeTag]);
        if( $locale == NULL ) {
            throw new Exception("Locale with code $localeTag not suported for interface");
        }

        $isLocaleSupported = false;
        foreach($this->adminService->getSupportedInterfaceLocales() as $supportedLocale) {
            if( $supportedLocale->getId() == $locale->getId() ) {
                $isLocaleSupported = true;
                break;
            }
        }
        if( !$isLocaleSupported ) {
            throw new Exception("Locale with code $localeTag not suported for interface");
        }
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->adminSelectedInterfaceLocaleSessionName] = $locale->getId();
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    public function computePasswordHash($password) {
        // use blowfish hash function with cost 10 ai. 10^10 iterations
        $options = [
            'cost' => 10,
        ];
        $hash = password_hash($password, PASSWORD_BCRYPT, $options);

        return $hash;
    }


    public function setLoggedUserInterfaceLocale() {
        /* @var $interfaceDbLocale DBLocale */
        $interfaceDbLocale = $this->getSelectedInterfaceLocale();
        $interfaceLocale = Locale::forLocaleTag($interfaceDbLocale->getLocaleTag());

        $this->gettextLocaleManager->setupLanguage($interfaceLocale);

        return $interfaceLocale;
    }

    function setRoleSupportedCountries(array $roleSupportedCountries = []) {
        // TODO: control format
        $this->roleSupportedCountries = $roleSupportedCountries;
    }

    protected function createAdminModuleInfo(AdminModule $adminModule, $localeId) {
        $moduleType = $adminModule->getType();
        $moduleActionsConfig = $this->getModuleActionsConfig($adminModule);

        $moduleActions = [];
        foreach($moduleActionsConfig['actions'] as $actionIdentifier => $actionConfig ) {
            $moduleAction = new AdminModuleActionInfo($actionConfig['name'], $actionIdentifier, $actionConfig['controllerId'], $actionConfig['methodName']);
            if( array_key_exists("parameters", $actionConfig) ) {
                $moduleAction->setParameters($actionConfig["parameters"]);
            } else {
                $moduleAction->setParameters([]);
            }
            if( array_key_exists("parentActionIdentifier", $actionConfig) ) {
                $moduleAction->setParentActionIdentifier($actionConfig["parentActionIdentifier"]);
            }
            $moduleActions[$actionIdentifier] = $moduleAction;
        }

        $moduleInfo = new AdminModuleInfo($adminModule, $moduleActions, $moduleActionsConfig["defaultAction"], $moduleActionsConfig["menuActions"], $localeId);

        // prefere name from module config
        if( array_key_exists("moduleName", $moduleActionsConfig) ) {
            $moduleInfo->setModuleName($moduleActionsConfig["moduleName"]);
        } else {
            $moduleInfo->setModuleName($adminModule->getName());
        }
        return $moduleInfo;
    }

    protected function getModuleActionsConfig(AdminModule $adminModule) {
        $moduleType = $adminModule->getType();
        if( array_key_exists($moduleType, $this->modulesTypesActionsCache) ) { // alredy cached
            return $this->modulesTypesActionsCache[$moduleType];
        }

        $moduleActions = [];
        $moduleLocation = $this->configDir ."/". $this->modulesDirs[$moduleType];
        if( file_exists($moduleLocation ."/". Config::MODULES_ACTION_CONFIG_FILE_NAME) ) {
            $moduleActions = require $moduleLocation ."/". Config::MODULES_ACTION_CONFIG_FILE_NAME;
        } else {
            $moduleActions = $this->modulesActions[$adminModule->getIdentifier()];
            foreach($moduleActions["actions"] as $actionIdentifier => $action) {
                $moduleActions["actions"][$actionIdentifier]["methodName"] = $action["action"];
            }
        }

        $this->modulesTypesActionsCache[$moduleType] = $moduleActions;

        return $moduleActions;
    }

    /**
     * @param null|string $universalPasswordHash
     */
    public function setUniversalPasswordHash(?string $universalPasswordHash = null): void {
        $this->universalPasswordHash = $universalPasswordHash;
    }
}
