<?php

namespace IZON\Admin\Domain;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\Mapping\Column;
use Doctrine\ORM\Mapping\Entity;
use Doctrine\ORM\Mapping\GeneratedValue;
use Doctrine\ORM\Mapping\Id;
use Doctrine\ORM\Mapping\JoinTable;
use Doctrine\ORM\Mapping\ManyToMany;
use Doctrine\ORM\Mapping\ManyToOne;
use Doctrine\ORM\Mapping\Table;
use IZON\Admin\Domain\AdminRole;
use IZON\Admin\Domain\Interfaces\AdminUserInterface;
use IZON\DBLocale\Domain\DBCountry;
use IZON\DBLocale\Domain\DBLocale;

/**
 * representation of admin user
 *
 * @Entity
 * @Table(name="core_admin_users")
 */
class AdminUser implements AdminUserInterface {

    /**
     * @Id
     * @Column
     * @GeneratedValue
     * @var int
     */
    protected $id;

    /**
     * @Column
     * @var string
     */
    protected $firstName;

    /**
     * @Column
     * @var string
     */
    protected $lastName;

    /**
     * @Column
     * @var string
     */
    protected $login;

    /**
     * @Column
     * @var string
     */
    protected $password;

    /**
     * @Column(type="boolean")
     * @var bool if to use module
     */
    protected $active = true;

    /**
     * @Column(type="boolean")
     * @var boolean is superuser, have access to all modules
     */
    protected $superuser = false;

    /**
     * @ManyToOne(targetEntity="\IZON\DBLocale\Domain\DBCountry")
     * @var DBCountry
     */
    protected $country;

    /**
     * @ManyToOne(targetEntity="\IZON\DBLocale\Domain\DBLocale", )
     * @var DBLocale
     */
    protected $interfaceLocale;

    /**
     * @ManyToMany(targetEntity="AdminRole", inversedBy="users")
     * @JoinTable(
     *     name="core_admin_users_admin_roles",
     *     joinColumns={@ORM\JoinColumn(name="fk_admin_user_id", referencedColumnName="id")},
     *     inverseJoinColumns={@ORM\JoinColumn(name="fk_admin_role_id", referencedColumnName="id")}
     * )
     * @var AdminRole[]|Collection
     */
    protected $roles;


    public function __construct() {
        $this->roles = new ArrayCollection();
    }


    /**
     * @inheritDoc
     */
    public function getRoles() : iterable {
      return $this->roles;
    }

    /**
     * @inheritDoc
     */
    public function setRoles(iterable $roles) : void {
        $this->roles = $roles;
    }

    /**
     * @inheritDoc
     */
    public function addRole(AdminRole $role) : void {
      $this->roles[$role->getId()] = $role;
    }

    /**
     * @inheritDoc
     */
    public function hasRoleByRole(AdminRole $role) : bool {
      if($this->superuser) {
          return true;
      }
      return $this->roles->contains($role);
    }

    /**
     * @inheritDoc
     */
    public function hasRoleByRoleId($roleId) : bool {
        if ($this->superuser) {
            return true;
        }
        $roles = $this->roles->map(function($role){ return $role->getId();})->toArray();
        //print_r($roles);die;
        if (!empty($roleId) && is_array($roleId)) {
            foreach ($roleId as $key => $value) {
                if (in_array($value, $roles)) {
                    return true;
                }
            }
        } else {
            return in_array($roleId, $roles);
        }
    }

    /**
     * @return int|null
     */
    public function getId(): ?int {
        return $this->id;
    }

    /**
     * @param int|null $id
     */
    public function setId(?int $id): void {
        $this->id = $id;
    }

    /**
     * @return null|string
     */
    public function getFirstName(): ?string {
        return $this->firstName;
    }

    /**
     * @param string $firstName
     */
    public function setFirstName(string $firstName): void {
        $this->firstName = $firstName;
    }

    /**
     * @return null|string
     */
    public function getLastName(): ?string {
        return $this->lastName;
    }

    /**
     * @param string $lastName
     */
    public function setLastName(string $lastName): void {
        $this->lastName = $lastName;
    }

    /**
     * @return null|string
     */
    public function getLogin(): ?string {
        return $this->login;
    }

    /**
     * @param string $login
     */
    public function setLogin(string $login): void {
        $this->login = $login;
    }

    /**
     * @return null|string
     */
    public function getPassword(): ?string {
        return $this->password;
    }

    /**
     * @param string $password
     */
    public function setPassword(string $password): void {
        $this->password = $password;
    }

    /**
     * @return bool
     */
    public function getActive(): bool {
        return $this->active;
    }

    /**
     * @param bool $active
     */
    public function setActive(bool $active): void {
        $this->active = $active;
    }

    /**
     * @return bool
     */
    public function getSuperuser(): bool {
        return $this->superuser;
    }

    /**
     * @param bool $superuser
     */
    public function setSuperuser(bool $superuser): void {
        $this->superuser = $superuser;
    }

    /**
     * @return DBCountry
     */
    public function getCountry(): ?DBCountry {
        return $this->country;
    }

    /**
     * @param DBCountry $country
     */
    public function setCountry(DBCountry $country): void {
        $this->country = $country;
    }

    /**
     * @return DBLocale
     */
    public function getInterfaceLocale(): ?DBLocale {
        return $this->interfaceLocale;
    }

    /**
     * @param DBLocale $interfaceLocale
     */
    public function setInterfaceLocale(DBLocale $interfaceLocale): void {
        $this->interfaceLocale = $interfaceLocale;
    }


}
