<?php

namespace IZON\Admin\Web\Controllers;

use Exception;
use IZON\Admin\Config;
use IZON\Admin\Domain\AdminUser;
use IZON\Admin\Services\LoggedAdminUserSessionServiceInterface;
use IZON\Forms\Form;
use IZON\MVC\Controller;
use IZON\MVC\HttpRequest;
use IZON\MVC\ModelAndView;
use function IZON\Admin\DI\moduleIdentifier;

/**
 *
 */
class LoginController implements Controller
{
    protected LoggedAdminUserSessionServiceInterface $sessionService;

    protected string $redirectControllerIdentifier;

    protected string $loginFormViewName = "admin/login";


    /**
     * @param LoggedAdminUserSessionServiceInterface $sessionService
     * @param string $redirectControllerIdentifier identifier of controller to redirect to after login
     */
    public function __construct(
        LoggedAdminUserSessionServiceInterface $sessionService,
        string $redirectControllerIdentifier
    ) {
        $this->sessionService = $sessionService;
        $this->redirectControllerIdentifier = $redirectControllerIdentifier;
    }

    public function login(HttpRequest $request)
    {
        $loginForm = $this->createLoginForm($request);
        $loginForm->setValues($request->getParameters());

        // neni vyplneny zadny login pravdepodobne je novy dotaz
        if ($loginForm->getField("login")->parse() == '') {
            $loginForm->getField('url')->serialize($request->getParameter('url'));
            $modelAndView = new ModelAndView($this->loginFormViewName, ["loginForm" => $loginForm]);
            return $modelAndView;
        }

        // pokusit se prihlasit
        try {
            $this->sessionService->loginUser(
                $loginForm->getField("login")->parse(),
                $loginForm->getField("password")->parse()
            );
            $url = $loginForm->getField('url')->parse();
            if (!empty($url)) {
                $modelAndView = ModelAndView::createURLRedirect($url);
            } else {
                $modelAndView = $this->redirectModule();
            }
            return $modelAndView;
        } catch (Exception $e) {
            if ($e->getCode() == 100) {
                $loginForm->getField('login')->addError(__("Pro login neexistuje uzivatel."));
            } elseif ($e->getCode() == 200) {
                $loginForm->getField('password')->addError(__("Pro login nesouhlasí heslo."));
            } elseif ($e->getCode() == 300) {
                $loginForm->getField('login')->addError(__("Pro login neexistuje uzivatel."));
            } else {
                $loginForm->addFormError($e->getMessage());
            }

            $modelAndView = new ModelAndView(
                $this->loginFormViewName,
                ["loginForm" => $loginForm, "loginError" => true]
            );
        }

        return $modelAndView;
    }

    protected function createLoginForm(HttpRequest $request)
    {
        $form = new Form();
        $form->addCharField("login", "Login");
        $form->addPasswordField("password", "Heslo");
        $form->addHiddenField('url');
        return $form;
    }

    /**
     *
     * @return ModelAndView
     */
    protected function redirectModule()
    {
        $loggedUser = $this->sessionService->getLoggedUser();
        return $this->redirectModuleConfig($loggedUser);
    }

    /**
     * get redirect from config
     * @param AdminUser $loggedUser
     * @return ModelAndView
     */
    protected function redirectModuleConfig(AdminUser $loggedUser)
    {
        try {
            $redirect = Config::getValue('adminLoginRedirect');
        } catch (Exception $ex) {
            return ModelAndView::createControllerRedirect($this->redirectControllerIdentifier, []);
        }
        $redirectConfig = null;
        // config for user
        if (isset($redirect['user'][$loggedUser->getId()])) {
            $redirectConfig = $redirect['user'][$loggedUser->getId()];
            if (!$loggedUser->getSuperuser()) {
                $hasAccess = false;
                // check permisions for module
                foreach ($loggedUser->getRoles() as $role) {
                    foreach ($role->getModules() as $module) {
                        if ($module->getIdentifier() == $redirectConfig['module']) {
                            $hasAccess = true;
                        }
                    }
                }
                // access denied
                if (!$hasAccess) {
                    $redirectConfig = null;
                }
            }
        }
        if (empty($redirectConfig)) {
            // role redirect config
            foreach ($loggedUser->getRoles() as $role) {
                if (isset($redirect['role'][$role->getId()])) {
                    $currentRedirect = $redirect['role'][$role->getId()];
                    if (empty($currentRedirect)) {
                        $redirectConfig = $currentRedirect;
                    } elseif ($currentRedirect['priority'] < $currentRedirect['priority']) {
                        $redirectConfig = $currentRedirect;
                    }
                }
            }
        }
        if (!empty($redirectConfig)) {
            return $this->createRedirect($redirectConfig);
        }
        return ModelAndView::createControllerRedirect($this->redirectControllerIdentifier, []);
    }

    protected function createRedirect($config)
    {
        $identifier = moduleIdentifier($config['controller'], $config['module']);
        $params = !empty($config['parameters']) ? $config['parameters'] : [];
        if (!isset($params['action'])) {
            $params['action'] = 'execute';
        }
        return new ModelAndView($identifier, $params, true);
    }
}
