<?php

namespace IZON\Admin\Web\Controllers;

use Exception;
use IZON\Admin\MVC\Context\AdminContext;
use IZON\Admin\Services\AbstractBaseService;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\Forms\Form;
use IZON\IO\File;
use IZON\IO\Image;
use IZON\MVC\HttpRequest;
use IZON\MVC\ModelAndView;

/**
 * @template BaseService of AbstractBaseService
 */
abstract class AbstractBaseController
{
    /**
     * @var BaseService
     */
    protected AbstractBaseService $service;

    /**
     * @var string v jakem adresari se nachazeji docasne soubory nahravane
     */
    protected string $filesFolder = "files";

    /**
     *
     * @var string preffix pro formulare kde je ukladaji obrazky
     */
    protected string $imagesPrefix = "images";

    /**
     *
     * @var string preffix pro formulare kde je ukladaji obrazky
     */
    protected string $filesPrefix = "files";

    /**
     * mineType to dataType mapping
     * @var array<string, string>
     */
    protected array $dataTypes = [
        'text/plain' => 'txt',
        'image/jpeg' => 'jpg',
        'application/zip' => 'zip',
        'video/mp4' => 'mp4',
        'video/mpeg' => 'mpeg',
        'video/avi' => 'mpeg',
        'application/mspowerpoint' => 'powerpoint',
        'application/excel' => 'excel',
        'application/x-excel' => 'excel',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'excel',
        'application/msword' => 'msword',
        'application/pdf' => 'pdf'
    ];


    /**
     * @param BaseService $service
     */
    public function __construct(AbstractBaseService $service)
    {
        $this->service = $service;
    }

    public function changeProperty(HttpRequest $request)
    {
        if ($request->hasParameter('propertyName')) {
            $this->service->toggleProperty($request->getParameter('id'), $request->getParameter('propertyName'));
        }

        $mav = ModelAndView::createControllerRedirect($request->getCalledControllerId(), ["action" => "execute"]); // presmerovavat na senam
        return $mav;
    }

    public function ajaxUploadFile(HttpRequest $request)
    {
        $id = $request->getParameter("id");
        if (!$id) {
            $id = 0;
        }

        // kde je soubor ulozen
        $fileURI = $_FILES["file"]["tmp_name"];
        // ciste jmeno
        $fileName = File::getSafeFileName($_FILES["file"]["name"]);
        $fileFolder = uniqid();

        // presun do temporary souboru
        // TODO: udelat nejak inteligentneji, aby se dano nastavovat
        $tempFileDir = $this->getFileUploadTempDir() . "/" . $fileFolder;
        if (!file_exists($tempFileDir)) {
            // vytvor temporry
            mkdir($tempFileDir, 0770, true);
        }
        $tempFileName = $tempFileDir . "/" . $fileName;
        move_uploaded_file($fileURI, $tempFileName);

        $mineType = mime_content_type($tempFileName);
        $dataType = $this->dataTypes[$mineType];

        $retval = ["id" => $fileFolder . "/" . $fileName,
            "fileName" => $fileName,
            "description" => '',
            "fileURL" => $fileFolder . "/" . $fileName,
            'mineType' => $mineType,
            'dataType' => $dataType
        ];

        $modelAndView = new ModelAndView("admin/simple/json-respose");
        $modelAndView->putParameter("json", $retval);

        return $modelAndView;
    }

    protected function getFileUploadTempDir()
    {
        return __BASE_DIR__ // @phpstan-ignore constant.notFound
            . "/app-data/tmp/user-data/" . session_id() . "/" . $this->filesFolder;
    }

    public function ajaxGetFileContent(HttpRequest $request)
    {
        $imgURL = $request->getParameter("imgURL");
        $file = $this->getFileUploadTempDir() . "/" . $imgURL;
        $image = new File($file);

        $modelAndView = new ModelAndView("admin/raw-data");
        $modelAndView->putParameter("contentType", $image->getMimeType());
        $modelAndView->putParameter("data", file_get_contents($file));

        return $modelAndView;
    }

    public function ajaxChangePositions(HttpRequest $request)
    {
        $response['result'] = 'error';
        if ($request->hasParameter('data')) {
            $data = $request->getParameter('data');
            foreach ($data as $key => $item) {
                if ($item != null || $item != '') {
                    $this->service->updateObjectPosition($key, $item);
                }
            }
            $response['result'] = 'ok';
        }
        $modelAndView = new ModelAndView("admin/raw-data");
        $modelAndView->putParameter("data", json_encode($response));

        return $modelAndView;
    }

    /**
     * Return object
     *
     * @param HttpRequest $request
     * @return object
     * @deprecated
     */
    public function _getBaseObject(HttpRequest $request)
    {
        if ($this->_isUpdateOperation($request)) {
            $id = $request->getParameter("id");
            $object = $this->service->get($id); // @phpstan-ignore method.notFound
        } else {
            $object = $this->createDomainObject(); // @phpstan-ignore method.notFound
        }
        return $object;
    }

    /**
     * Is posted opeation update?
     *
     * @param HttpRequest $request
     * @return boolean
     */
    public function _isUpdateOperation(HttpRequest $request)
    {
        return $request->hasParameter("id") && $request->getParameter("id") + 0 > 0 ? true : false;
    }

    public function ajaxGetCurrentPage(HttpRequest $request, AdminContext $context)
    {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/list/list-content"); // @phpstan-ignore property.notFound

        $paginatorConfig = $this->getPaginatorConfig($context);

        $this->saveFilter($context, $paginatorConfig);

        $modelAndView->putParameter("paginatorConfig", $paginatorConfig);
        $page = $this->getPage($paginatorConfig);
        $modelAndView->putParameter("pageContent", $page);

        return $modelAndView;
    }

    /**
     *
     * @param AdminContext $context
     * @return PaginatorConfig
     */
    protected function getPaginatorConfig(AdminContext $context)
    {
        $paginatorConfig = $context->getSessionParameter($this->getPaginatorSessionIndentifier() . "-filter");
        if ($paginatorConfig === null) {
            $paginatorConfig = $this->createPaginatorConfig();
        }
        return $paginatorConfig;
    }

    abstract protected function getPaginatorSessionIndentifier();

    abstract protected function createPaginatorConfig();

    protected function saveFilter(AdminContext $context, PaginatorConfig $paginatorConfig)
    {
        $session = $context->getSession();
        $session->lock();
        $session->setParameter($this->getPaginatorSessionIndentifier() . "-filter", $paginatorConfig);
        $session->setParameter($this->getPaginatorSessionIndentifier() . "-filter-active", true);
        $session->unlock();
    }

    abstract protected function getPage(PaginatorConfig $config);

    public function ajaxChangeListOrder(HttpRequest $request, AdminContext $context)
    {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/list/list-content"); // @phpstan-ignore property.notFound

        if (!$request->hasParameter("propertyName")) {
            throw new Exception("Neni nastaven parametr property");
        }
        $propertyName = $request->getParameter("propertyName");
        $paginatorConfig = $this->getPaginatorConfig($context);

        $order = $paginatorConfig->getOrders()[0];
        $paginatorConfig->clearOrders();
        $paginatorConfig->addOrder($propertyName, $order[0] == $propertyName ? !$order[1] : true);

        $this->saveFilter($context, $paginatorConfig);
        $modelAndView->putParameter("paginatorConfig", $paginatorConfig);
        $modelAndView->putParameter("pageContent", $this->getPage($paginatorConfig));

        return $modelAndView;
    }

    public function ajaxSetFilter(HttpRequest $request, AdminContext $context)
    {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/list/list-content"); // @phpstan-ignore property.notFound

        $filterForm = $this->createFilterForm();

        $paginatorConfig = $this->getPaginatorConfig($context);

        $filterForm->setValues($request->getParameters());
        $filterForm->flush($paginatorConfig);

        $this->modifyPaginator($paginatorConfig, $filterForm);

        // move to first page
        $paginatorConfig->setFirstResult(0);

        $this->saveFilter($context, $paginatorConfig);
        $modelAndView->putParameter("paginatorConfig", $paginatorConfig);
        $modelAndView->putParameter("pageContent", $this->getPage($paginatorConfig));

        return $modelAndView;
    }

    abstract protected function createFilterForm();

    abstract protected function modifyPaginator(PaginatorConfig $paginatorConfig, Form $filterForm);

    public function ajaxClearFilter(HttpRequest $request, AdminContext $context)
    {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/list/list-content"); // @phpstan-ignore property.notFound

        $this->clearFilter($context);

        $paginatorConfig = $this->getPaginatorConfig($context);
        $modelAndView->putParameter("paginatorConfig", $paginatorConfig);
        $modelAndView->putParameter("pageContent", $this->getPage($paginatorConfig));

        return $modelAndView;
    }

    protected function clearFilter(AdminContext $context)
    {
        $session = $context->getSession();
        $session->lock();
        $session->removeParameter($this->getPaginatorSessionIndentifier() . "-filter");
        $session->removeParameter($this->getPaginatorSessionIndentifier() . "-filter-active");
        $session->unlock();
    }

    public function ajaxSwirchPage(HttpRequest $request, AdminContext $context)
    {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/list/list-content"); // @phpstan-ignore property.notFound

        if (!$request->hasParameter("pageNumber")) {
            throw new Exception("Neni nastaven parametr property");
        }
        $pageNumber = $request->getParameter("pageNumber");

        $paginatorConfig = $this->getPaginatorConfig($context);
        $paginatorConfig->setFirstResult(($pageNumber - 1) * $paginatorConfig->getMaxPageSize());
        $this->saveFilter($context, $paginatorConfig);


        $modelAndView->putParameter("paginatorConfig", $paginatorConfig);
        $modelAndView->putParameter("pageContent", $this->getPage($paginatorConfig));

        return $modelAndView;
    }

    /**
     * smaze obsah adresare kompletne rekurzivne
     * @param string $dir
     */
    protected function deleteRecursivelly($dir = null)
    {
        if ($dir == null) {
            $dir = $this->getFileUploadTempDir();
        }
        if (!file_exists($dir)) {
            return;
        }
        foreach (scandir($dir) as $subFile) {
            $absFileDir = $dir . "/" . $subFile;

            if (is_dir($absFileDir)) {
                if ($subFile != "." && $subFile != "..") {
                    $this->deleteRecursivelly($absFileDir);
                    rmdir($absFileDir);
                }
            } else { // je soubor smazat ho
                unlink($absFileDir);
            }
        }
    }

    /**
     * Vratí pole obrazku které obdrzel v requestu - pouze gallery
     * @param HttpRequest $request
     * @param class-string $class tridy obrazku.
     * @return array
     */
    protected function getArrayOfImages(HttpRequest $request, $class)
    {
        $images = [];

        if ($request->hasParameter($this->imagesPrefix)) {
            $fileUploadTempDir = $this->getFileUploadTempDir();
            foreach ($request->getParameter($this->imagesPrefix) as $index => $value) {
                $image = new $class();
                $imgForm = $this->createImageForm([$this->imagesPrefix, $index]);
                $imgForm->setValues($request->getParameters());
                $imgForm->flush($image);

                //otoceni
                if (isset($value["turn"]) && $value["turn"] != 0) {
                    $image->setTurn($value["turn"]);
                }

                if ($image->getId() != 0) { // je updatovany obrazek
                    $images[] = $image;
                } else { // je nove nahravany obrazek
                    $tempFile = $fileUploadTempDir . "/" . $index;
                    if (is_file($tempFile)) {
                        $image->setImage(new Image($tempFile));
                        // TODO: pridat informace o obrazku co se ma vlozit

                        $images[] = $image;
                    }
                }
            }
        }

        return $images;
    }

    /**
     * Pouziva se pouze pro gallerie
     * @param array $prefix
     * @return Form
     */
    protected function createImageForm(array $prefix)
    {
        $form = new Form($prefix);
        $form->addIdField("id", __("Id"));
        $form->addIntegerField("position");
        $form->addHiddenField("photoFileName");
        $form->addHiddenField("newPhotoFileName");
        $form->addCharField('description');

        return $form;
    }

    /**
     * Vratí pole obrazku které obdrzel v requestu - pouze gallery
     * @param HttpRequest $request
     * @param class-string $class tridy obrazku.
     * @return array
     */
    protected function getArrayOfFiles(HttpRequest $request, $class)
    {
        $files = [];

        if ($request->hasParameter($this->filesPrefix)) {
            $fileUploadTempDir = $this->getFileUploadTempDir();
            foreach ($request->getParameter($this->filesPrefix) as $index => $value) {
                $file = new $class();
                $fileForm = $this->createFileForm([$this->filesPrefix, $index]);
                $fileForm->setValues($request->getParameters());
                $fileForm->flush($file);

                if ($file->getId() != 0) { // je updatovany soubor
                    $files[] = $file;
                } else { // je nove nahravany soubor
                    $tempFile = $fileUploadTempDir . "/" . $index;
                    if (is_file($tempFile)) {
                        $file->setFile(new File($tempFile));
                        // TODO: pridat informace o obrazku co se ma vlozit

                        $files[] = $file;
                    }
                }
            }
        }

        return $files;
    }

    /**
     * Pouziva se pouze pro soubory
     * @param array $prefix
     * @return Form
     */
    protected function createFileForm(array $prefix)
    {
        $form = new Form($prefix);
        $form->addIdField("id", __("Id"));
        $form->addIntegerField("position");
        $form->addCharField("name");
        $form->addHiddenField("newFileName");
        $form->addHiddenField("fileName");
        $form->addCharField('description');
        $form->addHiddenField('dataType')->setOneWay(true);

        return $form;
    }

    protected function getPhotoFilePath($fileName)
    {
        return $this->getFileUploadTempDir() . '/' . $fileName;
    }

    protected function getFilterActive(AdminContext $context)
    {
        return $context->getSessionParameter($this->getPaginatorSessionIndentifier() . "-filter-active");
    }
}
