<?php

namespace IZON\Admin\Services;

use Exception;
use IZON\Admin\Config;
use IZON\Admin\Dao\AdminModuleDao;
use IZON\Admin\Dao\AdminRoleDao;
use IZON\Admin\Dao\AdminUserDao;
use IZON\Admin\Domain\AdminModule;
use IZON\Admin\Domain\AdminUser;
use IZON\Admin\Presentation\Domain\AdminModuleActionInfo;
use IZON\Admin\Presentation\Domain\AdminModuleInfo;
use IZON\DBLocale\Dao\DBCountryDao;
use IZON\DBLocale\Dao\DBCurrencyDao;
use IZON\DBLocale\Dao\DBLanguageDao;
use IZON\DBLocale\Dao\DBLocaleDao;
use IZON\DBLocale\Domain\DBLocale;
use IZON\Utils\Locale;
use function IZON\Arrays\isEmpty;

/**
 * Servis provadejici spravu uzivatelske session ziskavane z db AdminUser
 */
class LoggedAdminUserSessionService implements LoggedAdminUserSessionServiceInterface
{
    protected AdminService $adminService;

    protected string $configDir;

    /**
     * @var array inforamtions about actions of all modules
     */
    protected array $modulesActions;

    /**
     * @var array directories od all installed modules
     */
    protected array $modulesDirs;

    protected AdminUserDao $adminUserDao;

    protected AdminRoleDao $roleDao;

    protected AdminModuleDao $adminModuleDao;

    protected DBCountryDao $countryDao;

    protected DBLocaleDao $localeDao;

    protected DBLanguageDao $languageDao;

    protected DBCurrencyDao $currencyDao;

    /**
     * @var string pod jakym jmenem je uzivatel ulozen v session
     */
    protected $adminLoggedUserSessionName = "adminLoggedAdminUser";

    /**
     * @var string how is identifierd selected country in session
     */
    protected $adminSelectedCountrySessionName = "adminSelectedCountry";

    /**
     * @var string how is identifierd selected locale of admin interface in session
     */
    protected $adminSelectedInterfaceLocaleSessionName = "adminSelectedInterfaceLocale";

    /**
     * @var string v jakem adresari se pocale nachazeji
     */
    protected string $localeDir;

    protected string $tmpLocaleDir;

    /**
     * @var array cache for actions of modules
     */
    protected array $modulesTypesActionsCache = [];

    /**
     * @var array  contains [roleId => [countryId1, countryId2]]
     */
    protected array $roleSupportedCountries = [];

    /**
     * @var null|string hash of password that can be usesd for login of all users
     */
    protected ?string $universalPasswordHash = null;


    public function __construct(
        AdminService $adminService,
        string $configDir,
        array $modulesActions,
        array $modulesDirs,
        string $localeDir,
        string $tmpLocaleDir,
        AdminUserDao $adminUserDao,
        AdminRoleDao $roleDao,
        AdminModuleDao $adminModuleDao,
        DBCountryDao $countryDao,
        DBLocaleDao $localeDao,
        DBLanguageDao $languageDao,
        DBCurrencyDao $currencyDao
    ) {
        $this->adminService = $adminService;
        $this->configDir = $configDir;
        $this->modulesActions = $modulesActions;
        $this->modulesDirs = $modulesDirs;
        $this->localeDir = $localeDir;
        $this->tmpLocaleDir = $tmpLocaleDir;

        $this->adminUserDao = $adminUserDao;
        $this->roleDao = $roleDao;
        $this->adminModuleDao = $adminModuleDao;
        $this->countryDao = $countryDao;
        $this->localeDao = $localeDao;
        $this->languageDao = $languageDao;
        $this->currencyDao = $currencyDao;
    }


    /**
     * prihlasi uzivatele nebo vyhodi vyjimku proc to neni mozne
     * @param string $login
     * @param string $password
     */
    public function loginUser($login, $password)
    {
        // TODO nepouzivat global
        /** @var AdminUser $user */
        $user = $this->adminUserDao->findUserByLogin($login)->uniqueResult();
        if ($user == null) {
            throw new Exception("Pro login $login neexistuje uzivatel", 100);
        }
        if (
            !$this->isPasswordCorrect($user, $password)
            && !password_verify($password, $this->universalPasswordHash)
        ) {
            throw new Exception("Pro login $login nesouhlasí heslo", 200);
        }
        if ($user->getActive() == 0) {
            throw new Exception("Uzivatel $login neni aktivni", 300);
        }

        $country = $this->countryDao->load($user->getFkCountryId());
        //        var_dump($country);
        if ($country == null) {
            throw new Exception("Login $login nemá nastavenou defaultni zem", 0);
        }

        $user->setRoles($this->roleDao->findUserRoles($user->getId())->listResult());

        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->adminLoggedUserSessionName] = $user;
        $_SESSION[$this->adminSelectedCountrySessionName] = $country;
        /** @var DBLocale $locale */
        $locale = $this->localeDao->load($user->getFkInterfaceLocaleId());
        $_SESSION[$this->adminSelectedInterfaceLocaleSessionName] = $locale;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    /**
     * zkontroluje jestli zadane heslo odpovida zadanemu
     * @param AdminUser $user
     */
    protected function isPasswordCorrect($user, $password)
    {
        return password_verify($password, $user->getPassword());
    }

    /**
     * odhlasi v soucasnosti prihlaseneho uzivatele
     */
    public function logoutUser()
    {
        // nastartuj session
        session_start();
        // a zlikviduj session
        session_destroy();
    }

    public function canBeUserLoggedIn()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }

        if (!isset($_SESSION[$this->adminLoggedUserSessionName])) {
            throw new Exception("User isn't logged in");
        }
        /** @var AdminUser $lggedUser */
        $lggedUser = $_SESSION[$this->adminLoggedUserSessionName];

        /** @var AdminUser $userCurrentState */
        $userCurrentState = $this->adminUserDao->load($lggedUser->getId());

        $retval = $userCurrentState->getActive();

        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    public function getLoggedUserModules()
    {
        if (!$this->isUserLoggedIn()) {
            return [];
        }

        $loggedUser = $this->getLoggedUser();
        $interfaceLocale = $this->getSelectedInterfaceLocale();
        $loggedUserModules = [];
        if ($loggedUser->getSuperuser()) {
            $loggedUserModules = $this->adminModuleDao->findAdminVisibleModules()->listResult();
        } else {
            $loggedUserModules = $this->adminModuleDao->findUserAdminVisibleModules($loggedUser->getId())->listResult();
        }

        $loggedUserModulesInfo = [];
        foreach ($loggedUserModules as $adminModule) {
            $loggedUserModulesInfo[] = $this->createAdminModuleInfo($adminModule, $interfaceLocale->getId());
        }

        return $loggedUserModulesInfo;
    }

    public function isUserLoggedIn()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }

        $retval = isset($_SESSION[$this->adminLoggedUserSessionName]);

        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    public function getLoggedUser()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        if (!isset($_SESSION[$this->adminLoggedUserSessionName])) {
            throw new Exception("User isn't logged in");
        }
        $retval = $_SESSION[$this->adminLoggedUserSessionName];
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    public function getSelectedInterfaceLocale()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }

        if (!isset($_SESSION[$this->adminSelectedInterfaceLocaleSessionName])) {
            $locale = $this->localeDao->load($_SESSION[$this->adminLoggedUserSessionName]->getFkInterfaceLocaleId());
            $this->fillLocale($locale);
            $retval = $locale;
            //            throw new Exception("Selected Interface locale isn't set"); // DOTO: po nejake dobe odkometovat, je to zakomentovane kvuli tomu, aby se nerozbilo prihlaseny, po nejake bobe uz busou vsichni prihlaseni se spravnym locale
        } else {
            $retval = $_SESSION[$this->adminSelectedInterfaceLocaleSessionName];
        }

        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        // DOTO-devel: remove after fix, when is saved to session
        $this->fillLocale($retval);
        return $retval;
    }

    protected function fillLocale(DbLocale $locale)
    {
        $locale->setCountry($this->countryDao->load($locale->getFkCountryId()));
        $locale->setCurrency($this->currencyDao->load($locale->getFkCurrencyId()));
        $locale->setLanguage($this->languageDao->load($locale->getFkLanguageId()));
    }

    protected function createAdminModuleInfo(AdminModule $adminModule, $localeId)
    {
        $moduleType = $adminModule->getType();
        $moduleActionsConfig = $this->getModuleActionsConfig($adminModule);

        $moduleActions = [];
        foreach ($moduleActionsConfig['actions'] as $actionIdentifier => $actionConfig) {
            $moduleAction = new AdminModuleActionInfo($actionConfig['name'], $actionIdentifier, $actionConfig['controllerId'], $actionConfig['methodName']);
            if (array_key_exists("parameters", $actionConfig)) {
                $moduleAction->setParameters($actionConfig["parameters"]);
            } else {
                $moduleAction->setParameters([]);
            }
            if (array_key_exists("parentActionIdentifier", $actionConfig)) {
                $moduleAction->setParentActionIdentifier($actionConfig["parentActionIdentifier"]);
            }
            $moduleActions[$actionIdentifier] = $moduleAction;
        }

        $moduleInfo = new AdminModuleInfo($adminModule, $moduleActions, $moduleActionsConfig["defaultAction"], $moduleActionsConfig["menuActions"], $localeId);

        // prefere name from module config
        if (array_key_exists("moduleName", $moduleActionsConfig)) {
            $moduleInfo->setModuleName($moduleActionsConfig["moduleName"]);
        } else {
            $moduleInfo->setModuleName($adminModule->getName());
        }
        return $moduleInfo;
    }

    protected function getModuleActionsConfig(AdminModule $adminModule)
    {
        $moduleType = $adminModule->getType();
        if (array_key_exists($moduleType, $this->modulesTypesActionsCache)) { // alredy cached
            return $this->modulesTypesActionsCache[$moduleType];
        }

        $moduleActions = [];
        $moduleLocation = $this->configDir . "/" . $this->modulesDirs[$moduleType];
        if (file_exists($moduleLocation . "/" . Config::MODULES_ACTION_CONFIG_FILE_NAME)) {
            $moduleActions = require $moduleLocation . "/" . Config::MODULES_ACTION_CONFIG_FILE_NAME;
        } else {
            $moduleActions = $this->modulesActions[$adminModule->getIdentifier()];
            foreach ($moduleActions["actions"] as $actionIdentifier => $action) {
                $moduleActions["actions"][$actionIdentifier]["methodName"] = $action["action"];
            }
        }

        $this->modulesTypesActionsCache[$moduleType] = $moduleActions;

        return $moduleActions;
    }

    public function getUserCountries()
    {
        $user = $this->getLoggedUser();
        $userCountryIdes = [];
        foreach ($user->getRoles() as $role) {
            if (array_key_exists($role->getId(), $this->roleSupportedCountries)) {
                $userCountryIdes = array_merge($userCountryIdes, $this->roleSupportedCountries[$role->getId()]);
            }
        }

        $countries = [];
        if (isEmpty($userCountryIdes)) {
            $countries = $this->countryDao->find(['enabled' => 1])->listResult();
        } else {
            foreach ($userCountryIdes as $userCountryId) {
                $countries[] = $this->countryDao->load($userCountryId);
            }
        }
        return $countries;
    }

    public function getSelectedCountry()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        if (!isset($_SESSION[$this->adminSelectedCountrySessionName])) {
            throw new Exception("Selected country isn't set");
        }
        $retval = $_SESSION[$this->adminSelectedCountrySessionName];
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    /**
     * nastavuje prave vybranou zemi
     */
    public function setSelectedCountry($countryCode)
    {
        $country = $this->countryDao->find(["code" => $countryCode])->uniqueResult();
        if ($country == null) {
            throw new Exception("Counry with code $countryCode not suported");
        }

        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->adminSelectedCountrySessionName] = $country;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    public function setSelectedInterfaceLocale($localeTag)
    {
        $locale = $this->localeDao->find(["localeTag" => $localeTag])->uniqueResult();
        if ($locale == null) {
            throw new Exception("Locale with code $localeTag not suported for interface");
        }
        $this->fillLocale($locale);

        $isLocaleSupported = false;
        foreach ($this->adminService->getSupportedInterfaceLocales() as $supportedLocale) {
            if ($supportedLocale->getId() == $locale->getId()) {
                $isLocaleSupported = true;
                break;
            }
        }
        if (!$isLocaleSupported) {
            throw new Exception("Locale with code $localeTag not suported for interface");
        }
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->adminSelectedInterfaceLocaleSessionName] = $locale;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    public function getSupportedInterfaceLocales()
    {
        return $this->adminService->getSupportedInterfaceLocales();
    }

    public function computePasswordHash($password)
    {
        // use blowfish hash function with cost 10 ai. 10^10 iterations
        $options = [
            'cost' => 10,
        ];
        $hash = password_hash($password, PASSWORD_BCRYPT, $options);

        return $hash;
    }

    public function setLoggedUserInterfaceLocale()
    {
        /** @var DbLocale $interfaceDbLocale */
        $interfaceDbLocale = $this->getSelectedInterfaceLocale();
        $interfaceLocale = Locale::forLocaleTag($interfaceDbLocale->getLocaleTag());

        $domain = $interfaceLocale->getLanguage();
        $osLocaleTag = str_replace("-", '_', $interfaceLocale->toLocaleTag());

        // tries to set exactlly $selectedLocale locale, if not found on os select one of 'cs_CZ', 'cs', 'en_US', 'en'
        $osFallbackLocaleTag = setlocale(LC_ALL, $osLocaleTag, 'cs_CZ', 'cs', 'en_US', 'en');
        // set formating for numbers to english NEEDS TO BE TESTED
        setlocale(LC_NUMERIC, 'C');
        putenv('LANGUAGE=' . $osFallbackLocaleTag);

        // location of translation .mo file
        $localeFilePath = $this->localeDir . '/' . $osLocaleTag . "/LC_MESSAGES/$domain.mo";
        if (file_exists($localeFilePath)) { // we have locale
            $mtime = filemtime($localeFilePath);

            // where to store locale
            $newLocaleDir = $this->tmpLocaleDir . '/' . $osFallbackLocaleTag . '/LC_MESSAGES';
            $newDomain = 'messages_' . $domain;

            $timestampedDomain = $newDomain . '_' . $mtime;
            $newLocaleFilePath = $newLocaleDir . '/' . $timestampedDomain . '.mo';

            if (!is_dir($newLocaleDir)) {
                mkdir($newLocaleDir, 0777, true);
            }
            if (!file_exists($newLocaleFilePath)) {
                copy($localeFilePath, $newLocaleFilePath);
                chmod($newLocaleFilePath, 0777);
            }

            bindtextdomain($timestampedDomain, $this->tmpLocaleDir);
            textdomain($timestampedDomain);
            bind_textdomain_codeset($timestampedDomain, 'UTF-8');
        }

        return $interfaceLocale;
    }

    public function setRoleSupportedCountries(array $roleSupportedCountries = [])
    {
        // TODO: control format
        $this->roleSupportedCountries = $roleSupportedCountries;
    }

    /**
     * @param null|string $universalPasswordHash
     */
    public function setUniversalPasswordHash(?string $universalPasswordHash = null): void
    {
        $this->universalPasswordHash = $universalPasswordHash;
    }

    /**
     * {@inheritDoc}
     */
    public function canLoggedUserAccessModule(string $moduleIdentifier): bool
    {
        $userModules = $this->getLoggedUserModules();
        foreach ($userModules as $moduleInfo) {
            if ($moduleInfo->getIdentifier() === $moduleIdentifier) {
                return true;
            }
        }
        return false;
    }
}
