<?php

namespace IZON\Admin\Services\Impl;

use IZON\Admin\Dao\AdminModuleDao;
use IZON\Admin\Dao\AdminModuleSettingDao;
use IZON\Admin\Domain\AdminModule;
use IZON\Admin\Domain\AdminModuleSetting;
use IZON\Admin\Services\AdminService;
use IZON\DBLocale\Dao\DBCountryDao;
use IZON\DBLocale\Dao\DBCurrencyDao;
use IZON\DBLocale\Dao\DBLanguageDao;
use IZON\DBLocale\Dao\DBLocaleDao;
use IZON\DBLocale\Domain\DBLocale;
use function json_decode;

class AdminServiceImpl implements AdminService
{
    /**
     * @var array<int>
     */
    protected array $supportedInterfaceLocaleIdes;

    protected AdminModuleDao $adminModuleDao;

    protected AdminModuleSettingDao $adminModuleSettingDao;

    protected DBLocaleDao $localeDao;

    protected DBCountryDao $countryDao;

    protected DBLanguageDao $languageDao;

    protected DBCurrencyDao $currencyDao;

    public function __construct(
        AdminModuleDao $adminModuleDao,
        AdminModuleSettingDao $adminModuleSettingDao,
        DBLocaleDao $localeDao,
        DBCountryDao $countryDao,
        DBLanguageDao $languageDao,
        DBCurrencyDao $currencyDao
    ) {
        $this->adminModuleDao = $adminModuleDao;
        $this->adminModuleSettingDao = $adminModuleSettingDao;
        $this->localeDao = $localeDao;
        $this->countryDao = $countryDao;
        $this->languageDao = $languageDao;
        $this->currencyDao = $currencyDao;
    }

    /**
     * {@inheritdoc}
     */
    public function getActiveModules()
    {
        return $this->adminModuleDao->findActiveModules()->listResult();
    }

    /**
     * {@inheritdoc}
     */
    public function getModuleSetting($moduleIdentifier, $parameterIdentifier)
    {
        /** @var AdminModule $module */
        $module = $this->adminModuleDao->find(["identifier" => $moduleIdentifier])->uniqueResult();
        /** @var AdminModuleSetting $moduleSettings */
        $moduleSettings = $this->adminModuleSettingDao
            ->find(["fkAdminModuleId" => $module->getId(), "paramIdentifier" => $parameterIdentifier])
            ->uniqueResult();
        return $moduleSettings;
    }

    /**
     * {@inheritdoc}
     */
    public function getModuleSettings($moduleIdentifier)
    {
        /** @var AdminModule $module */
        $module = $this->adminModuleDao->find(["identifier" => $moduleIdentifier])->uniqueResult();
        return $this->adminModuleSettingDao->find(["fkAdminModuleId" => $module->getId()])->listResult();
    }

    /**
     * {@inheritdoc}
     */
    public function getSupportedCountries()
    {
        return $this->countryDao->find(['enabled' => 1])->listResult();
    }

    /**
     * {@inheritdoc}
     */
    public function getSupportedInterfaceLocales()
    {
        $supportedInterfaceLocales = [];
        foreach ($this->supportedInterfaceLocaleIdes as $supportedInterfaceLocaleId) {
            $locale = $this->localeDao->load($supportedInterfaceLocaleId);
            $country = $this->countryDao->load($locale->getFkCountryId());
            $locale->setCountry($country);
            $language = $this->languageDao->load($locale->getFkLanguageId());
            $locale->setLanguage($language);
            $supportedInterfaceLocales[] = $locale;
        }
        return $supportedInterfaceLocales;
    }

    /**
     * {@inheritdoc}
     */
    public function getAdminVersion()
    {
        $version = json_decode(
            file_get_contents(dirname(__DIR__, 3) . '/composer.json'),
            true
        )['version'];
        return $version;
    }

    /**
     * {@inheritdoc}
     */
    public function setSupportedInterfaceLocaleIdes(array $supportedInterfaceLocaleIdes = [1])
    {
        $this->supportedInterfaceLocaleIdes = $supportedInterfaceLocaleIdes;
    }

    protected function fillLocale(DBLocale $locale)
    {
        $country = $this->countryDao->load($locale->getFkCountryId());
        $locale->setCountry($country);
        $language = $this->languageDao->load($locale->getFkLanguageId());
        $locale->setLanguage($language);
        $currency = $this->currencyDao->load($locale->getFkCurrencyId());
        $locale->setCurrency($currency);
    }
}
