<?php

namespace IZON\Admin\Services;

use Exception;
use IZON\DB\Dao;
use IZON\DB\DBObject;
use IZON\DB\OrderBy;
use IZON\Utils\Slug;

/**
 * Base servis pro administraci - obsahuje metody vyuzivane AbstractBaseControllerem
 * Obsahuje metody pro changeSimpleProperty
 * dale metody pro praci s obrazky standartaně v db vazba 1:N
 *
 * @template E of DBObject
 * @template EDao of Dao<E>
 */
abstract class AbstractBaseService
{
    /**
     * Standartni dao pro change simple property true false
     * @var null|EDao
     */
    public ?Dao $dao = null;

    /**
     * Standartní dao pro manipulaci z obrazky
     * @var null|Dao
     */
    public ?Dao $imageDao = null;


    /**
     * @param EDao $dao
     */
    public function __construct(Dao $dao)
    {
        $this->dao = $dao;
    }

    /**
     * Slouzi k jednocuche zmene boolean vlastnosti
     * Nejcasteji zmena aktivity zaznamu
     * If is now any active transaction then use this transaction else make own transaction
     * @param int|string $objectId - id of object to toggle property on
     * @param string $propertyName - name of boolean property (attribute),
     */
    public function toggleProperty($objectId, string $propertyName): bool
    {
        $dao = $this->getDao();

        $object = $dao->load($objectId);

        $getterGet = 'get' . ucfirst($propertyName);
        $getterIs = 'is' . ucfirst($propertyName);
        $setter = 'set' . ucfirst($propertyName);
        $method = null;

        if (method_exists($object, $getterGet)) {
            $method = $getterGet;
        } elseif (method_exists($object, $getterIs)) {
            $method = $getterIs;
        } else {
            throw new Exception('Class: ' . get_class($object) . ' is not have method ' . $getterIs . ' or ' . $getterGet);
        }

        if (method_exists($object, $setter)) {
            $result = !$object->$method();
            $object->$setter($result);
            $dao->update($object);
        } else {
            throw new Exception('Class: ' . get_class($object) . ' is nat have method  ' . $setter);
        }
        return $result;
    }

    /**
     * @return EDao dao for service main object
     */
    protected function getDao(): Dao
    {
        if (empty($this->dao)) {
            throw new Exception('Atribute $this->dao is not set');
        }
        return $this->dao;
    }

    /**
     * Slouzi k jednocuche zmene boolean vlastnosti
     * Nejcasteji zmena aktivity zaznamu
     * If is now any active transaction then use this transaction else make own transaction
     * @param int|string $objectId - id of updated object
     * @param string $propertyName - name of boolean property (atribute),
     * @deprecated use toggleProperty()
     */
    public function changeSimpleProperty($objectId, $propertyName)
    {
        $this->toggleProperty($objectId, $propertyName);
    }

    /********************** IMAGE methods ****************************************/
    /*****************************************************************************/

    /**
     * Update pozice objektu
     * Update position of object
     * @param int $objId - id of object which you want to update, object have to contain atribute position and method set Position()
     * @param int $position - position want i set
     */
    public function updateObjectPosition($objId, $position)
    {
        $dao = $this->getDao();
        $obj = $dao->load($objId);
        $obj->setPosition($position); // @phpstan-ignore method.notFound
        $dao->update($obj);
    }

    /**
     * Metoda pro ukladani std. obrazku - save/update
     * !!  Doporučuji tuto metodu obalit transakci !!!
     * @param DBObject $obj - $object which contain images
     * @param string $methodName - name of method to get images from $obj
     * @param string $connPropName property name that realises connection to $obj
     */
    public function updateImages(DBObject $obj, $methodName = 'getImages', $connPropName = "fkObjectId")
    {
        $imageDao = $this->getImageDao();
        // zpracovani obrazku
        $images = $imageDao->find([$connPropName => $obj->getId()])->listResult(); // @phpstan-ignore method.notFound
        $imagesHash = [];
        foreach ($images as $image) {
            $imagesHash[$image->getId()] = $image; // @phpstan-ignore method.notFound
        }

        foreach ($obj->$methodName() as $newImage) {
            //otoceni
            $turn = 0;
            if ($newImage->getTurn() != null && $newImage->getTurn() != 0) {
                $turn = $newImage->getTurn();
            }

            $connectionSetterName = "set" . ucfirst($connPropName);
            $newImage->$connectionSetterName($obj->getId()); // @phpstan-ignore method.notFound

            if (isset($imagesHash[$newImage->getId()])) {
                $oldImage = $imagesHash[$newImage->getId()];
                $newImage->setImage($oldImage->getImage()); // @phpstan-ignore method.notFound
                $imageDao->update($newImage);
                unset($imagesHash[$newImage->getId()]);
            } else {
                $imageDao->save($newImage);
            }

            if ($turn != 0) {
                $newImage->setTurn($turn);
                $this->turnImage($newImage);
            }
        }
        // odstranit smazane
        foreach ($imagesHash as $image) {
            $imageDao->delete($image);
        }
    }

    /**
     * @return Dao dao for image object
     */
    protected function getImageDao(): Dao
    {
        if (empty($this->imageDao)) {
            throw new Exception('Atribute $this->imageDao is not set');
        }
        return $this->imageDao;
    }

    /**
     * Otoci obrazek
     * Turn image which DBobject contain
     * @param DBObject $image - object, ktery obsahuje obrazek ( DB object musi obsahovat atribut turn a metodu getTurn())
     * @param string $method - nazev metody ktere vrati obrazek z DBObjektu
     */
    public function turnImage(DBObject $image, $method = 'getImage')
    {
        if ($image->getTurn() != 0 && $image->getTurn() != '') { // @phpstan-ignore-line
            $picture = $image->$method();
            $path = ($picture->getFsPath());

            //nacteni
            $mine = $picture->getMimeType();
            switch ($mine) {
                case "image/gif":
                    $img = imagecreatefromgif($path);
                    break;
                case "image/jpeg":
                    $img = imagecreatefromjpeg($path);
                    break;
                case "image/png":
                    $img = imagecreatefrompng($path);
                    imagealphablending($img, false);
                    imagesavealpha($img, true);
                    break;
                default:
                    throw new Exception('Unknown image type.');
            }
            //rotace
            $rotate = imagerotate($img, $image->getTurn(), 0); // @phpstan-ignore method.notFound

            //ulozeni
            switch ($mine) {
                case "image/gif":
                    imagegif($rotate, $path);
                    break;
                case "image/jpeg":
                    imagejpeg($rotate, $path);
                    break;
                case "image/png":
                    imagealphablending($rotate, false);
                    imagesavealpha($rotate, true);
                    imagepng($rotate, $path);
                    break;
            }

            imagedestroy($img);
            imagedestroy($rotate);
        }
    }

    /**
     * Smaže všechny obrazky k danému objektu
     *  * !!  Doporučuji tuto metodu obalit transakci !!!
     * @param int $objectId - id objektu ke smazani
     * @param string $connPropName property name that realises connection to $obj
     */
    public function deleteAllImages($objectId, $connPropName = "fkObjectId")
    {
        $imageDao = $this->getImageDao();
        $images = $imageDao->find([$connPropName => $objectId])->listResult();
        foreach ($images as $image) {
            $imageDao->delete($image);
        }
    }

    /**
     * Return images by fk, in order by column position
     * Method try to order by position on failld return order by default
     * @param int|string $fkObjId ID of FK key
     * @param null|string $tableName Name of images table, Have not be set TODO table name isn't needed any more, mabe remove in future?
     * @param string $connPropName
     * @return array Array of images
     */
    public function getImagesByFkId(
        $fkObjId,
        $tableName = null,
        $connPropName = "fkObjectId"
    ) {
        $imageDao = $this->getImageDao();
        try {
            $images = $imageDao->find([$connPropName => $fkObjId], ['position' => OrderBy::ASC])->listResult();
        } catch (Exception $e) {
            $images = $imageDao->find([$connPropName => $fkObjId])->listResult();
        }

        return $images;
    }


    /************* PROTECTED and PRIVATE method *******************************/

    /**
     * simple Update (or save) objects in relation 1 to N to object
     * Require every object have getter - getId(), that return id of object
     *
     * Not use to exdens object to update (complicated updates which change more columns than only foreignKey colum)
     * Method is not use transaction
     *
     * Setters of columns(atributtes) have to have same name as column Name (ex. column - fkArticleId, getter - setFkArticleId)
     *
     * @param int $id id of main object
     * @param Dao $dao dao of ralationship objects
     * @param iterable<DBObject> $objects array of relationship objects
     * @param string $columnName name of column in objects with uset as foreign key, in lower-CamelCase
     * @param null|callable $beforeDeleteCallable is called before delete old item
     * @param bool $delete delete (true), update(false) old item
     */
    public function update1toNObject(
        $id,
        Dao $dao,
        iterable $objects,
        string $columnName = 'fkObjectId',
        $beforeDeleteCallable = null,
        bool $delete = true
    ) {
        if (empty($objects)) {
            $objects = [];
        }
        $setter = 'set' . ucfirst($columnName);

        $toUpdate = [];
        $toSave = [];
        $toDelete = [];

        //process update
        $olds = $dao->find([$columnName => $id])->listResult();
        $oldsHash = [];
        foreach ($olds as $old) {
            $oldsHash[$old->getId()] = $old; // @phpstan-ignore method.notFound
        }
        foreach ($objects as $object) {
            $object->$setter($id);
            if (isset($oldsHash[$object->getId()])) { // @phpstan-ignore method.notFound
                $toUpdate[] = $object;
                unset($oldsHash[$object->getId()]); // @phpstan-ignore method.notFound
            } else {
                $toSave[] = $object;
            }
        }
        foreach ($oldsHash as $old) {
            if (!empty($beforeDeleteCallable) && is_callable($beforeDeleteCallable)) { // @phpstan-ignore function.alreadyNarrowedType
                call_user_func($beforeDeleteCallable, $old);
            }
            if ($delete) {
                $toDelete[] = $old;
            } else {
                $toUpdate[] = $old;
            }
        }

        foreach ($toDelete as $item) {
            $dao->delete($item);
        }
        foreach ($toUpdate as $item) {
            $dao->update($item);
        }
        foreach ($toSave as $item) {
            $dao->save($item);
        }
    }

    /**
     * simple update (or save) objects in relation N to M
     * Require every object have getter - getId(), that return id of object
     *
     * Not use to exdens object to update - use onli to simple M to N with table that contain only 3 columns (id, fkMandoryId, fkSecondaryId)
     * Method is not use transaction
     *
     * Gettes and setters of columns(atributtes) have to have same name as column Name (ex. column - fkArticleId, getter - getFkArticleId)
     *
     * @param int|string $id of mandory object
     * @param Dao $daoRelationShip dao of relationTable
     * @param iterable<DBObject> $objects array of secondary objects
     * @param string $mandatoryColumnName mandory column name in relation table
     * @param string $secondaryColumnName secondary column name in realtion table
     * @param class-string $relationDomain class name of relation domain
     */
    public function updateNtoNObject(
        $id,
        Dao $daoRelationShip,
        iterable $objects,
        string $mandatoryColumnName,
        string $secondaryColumnName,
        string $relationDomain
    ) {
        //make setters,getters
        $mandatoryAtributeSetter = 'set' . ucfirst($mandatoryColumnName);
        $relationAttributeGetter = 'get' . ucfirst($secondaryColumnName);
        $relationAttributeSetter = 'set' . ucfirst($secondaryColumnName);

        //get old records
        $olds = $daoRelationShip->find([$mandatoryColumnName => $id])->listResult();

        //TODo: check if getters and setter exist

        //process update
        $oldsHash = [];
        foreach ($olds as $old) {
            $oldsHash[$old->$relationAttributeGetter()] = $old;
        }
        foreach ($objects as $object) {
            if (isset($oldsHash[$object->getId()])) { // @phpstan-ignore method.notFound
                unset($oldsHash[$object->getId()]); // @phpstan-ignore method.notFound
            } else {
                $relation = new $relationDomain();
                $relation->$mandatoryAtributeSetter($id);
                $relation->$relationAttributeSetter($object->getId()); // @phpstan-ignore method.notFound

                $daoRelationShip->save($relation);
            }
        }
        foreach ($oldsHash as $old) {
            $daoRelationShip->delete($old);
        }
    }

    /**
     * Make unique slug in table
     * @param string $name string from which I make slug
     * @param Dao $dao Dao class of table
     * @param string $columnName column where is save slug string
     * @return string Slug
     */
    protected function createUniqueSlugInTable($name, Dao $dao, string $columnName = 'slug')
    {
        $slug = Slug::createSlug($name);
        $res = $dao->find([$columnName => $slug])->listResult();
        if (!empty($res)) {
            $num = 0;
            do {
                $num++;
                $res = $dao->find([$columnName => $slug . '-' . $num])->listResult();
            } while (!empty($res));
            $slug = $slug . '-' . $num;
        }

        return $slug;
    }
}
