<?php

namespace IZON\Admin\MVC\Routers;

use Exception;
use IZON\Admin\Config;

use IZON\Admin\MVC\Exceptions\AdminPageNotFoundException;

use IZON\Admin\MVC\ModuleInfo;

use IZON\DB\DBConnection;

use IZON\Logs\Logger;
use IZON\MVC\HttpRequest;
use IZON\MVC\Routers\RouteDefinition;

use IZON\MVC\Routers\RouteInfo;
use IZON\Utils\Locale;
use PDO;

/**
 * Definice routovani, ktera provadi routovani pro jednotlive moduly admina
 */
class AdminRouteDefinition implements RouteDefinition
{
    /**
     * prefix pro routovani admina
     * @var type
     */
    protected $adminRoutePattern = "^/admin";

    /**
     * specificke routovani primo pro prihlaseni a odhlaseni z adminu
     * neaplikuji se na ne $adminInterceptorIdentifiers
     * @var array
     */
    protected $adminLoginLogouRoutes = [
        "^/admin/login" => [Config::LOGIN_CONTROLLER_IDENTIFIER, "login"],
        "^/admin/logout" => [Config::LOGOUT_CONTROLLER_IDENTIFIER, "logout"]
    ];

    /**
     * routovani specificke v ramci admina
     */
    protected $adminSpecificRoutes = [
        "^/admin/dashboard" => [Config::DASHBOARD_CONTROLLER_IDENTIFIER],
    ];

    /**
     * @var array interceproty, ktere se provadeji pro kazdy conroller admina
     */
    protected $adminInterceptorIdentifiers = [
        Config::LOAD_ADMIN_VERSION_INTERCEPTOR_IDENTIFIER
    ];

    /**
     * @var array interceptory provadene kolem modulovych controlleru
     */
    protected $adminInternalInterceptorIdentifiers = [
        Config::LOGGED_USER_CHECK_INTERCEPTOR_IDENTIFIER,
        Config::LOGGED_USER_INTERCEPTOR_IDENTIFIER,
        Config::MODULES_ACTIONS_INTERCEPTOR_IDENTIFIER
    ];

    /**
     * @var DBConnection
     */
    protected $dbConnection;

    /**
     * logger pro tridu
     * @var Logger
     */
    protected $log;

    public function __construct($dbConnection)
    {
        $this->dbConnection = $dbConnection;
        // logovani
        $this->log = Logger::getLogger(__CLASS__);
    }

    public function findRoute(HttpRequest $request)
    {
        $this->log->info("Trying to find routing for url: " . $request->getUrl());

        // projit specialni adresy admina, hlavne prihlaseni a odhlaseni
        foreach ($this->adminLoginLogouRoutes as $url => $rule) {
            if (preg_match("#" . $url . "#", $request->getUrl())) {
                $routeInfo = new RouteInfo();
                $routeInfo->setUrl($request->getUrl());
                $routeInfo->setPattern($url);
                $routeInfo->setContrlollerId($rule[0]);
                $routeInfo->setMethodName($rule[1]);
                $routeInfo->setInterceptors($this->adminInterceptorIdentifiers);

                $this->log->info("Calling of login or logout url: " . $url);

                return $routeInfo;
            }
        }

        // projde mapovani adres pro admin modul
        // hlavne dashboard, na ktery se presmerovava po prihlaseni
        foreach ($this->adminSpecificRoutes as $url => $rule) {
            if (preg_match("#" . $url . "#", $request->getUrl())) {
                $routeInfo = new RouteInfo();
                $routeInfo->setUrl($request->getUrl());
                $routeInfo->setPattern($url);
                $routeInfo->setContrlollerId($rule[0]);
                $routeInfo->setMethodName(null);
                $routeInfo->setInterceptors(array_merge($this->adminInterceptorIdentifiers, $this->adminInternalInterceptorIdentifiers));

                $this->log->info("Calling of default admin routes: " . $url);

                return $routeInfo;
            }
        }

        // je adresa modulu adminu
        if (preg_match("#" . $this->adminRoutePattern . "#", $request->getUrl())) { // je url admina
            $this->log->info("Calling admin route: " . $request->getUrl());

            $modulePattern = $this->adminRoutePattern . '/([a-zA-Z0-9/-]+)/([a-zA-Z0-9\.]+)/([a-zA-Z0-9]*)';
            if (preg_match('#' . $modulePattern . '#', $request->getUrl(), $matches)) {  // reqexp musi byt obalen delimiterem
                $moduleIdentifier = $matches[1];
                $bareControllerId = $matches[2];
                $methodName = $matches[3];

                if ($methodName == '') {
                    $methodName = "execute";
                }
            } elseif (preg_match("#" . $this->adminRoutePattern . "/{0,1}#", $request->getUrl())) { // vola se adresa /admin
                // presmeruj na dashboard
                $routeInfo = new RouteInfo();
                $routeInfo->setUrl($request->getUrl());
                $routeInfo->setPattern($url);
                $routeInfo->setContrlollerId(Config::DASHBOARD_CONTROLLER_IDENTIFIER);
                $routeInfo->setMethodName(null);
                $routeInfo->setInterceptors(array_merge($this->adminInterceptorIdentifiers, $this->adminInternalInterceptorIdentifiers));

                $this->log->info("Calling of default admin routes: " . $url);

                return $routeInfo;
            } else {
                throw new AdminPageNotFoundException("Request url " . $request->getUrl() . " doesn't match pattern module $modulePattern");
            }

            //            $moduleIdentifier = $request->getParameter("moduleIdentifier");
            //            $bareControllerId = $request->getParameter("controllerId");
            //            $methodName = $request->getParameter("methodName");

            //            if( $moduleIdentifier == '' ) { // moduleId not set  ???? redirect to dashboard
            //                $routeInfo = new RouteInfo();
            //                $routeInfo->setUrl($request->getUrl());
            ////                $routeInfo->set

            // TODO: pouzit neco univerzalnejsiho
            /** @var \PDO $conn */
            $conn = $this->dbConnection->getPDO();
            $sql = 'select m.*
                from admin_modules m
                where m.identifier = :moduleIdentifier';
            $statement = $conn->prepare($sql);
            $statement->bindValue(":moduleIdentifier", $moduleIdentifier);
            $statement->execute();

            if (!$row = $statement->fetch(PDO::FETCH_ASSOC)) {
                $e = new Exception("Calling to nonexisting module $moduleIdentifier and modules controller $bareControllerId");
                $this->log->warning("Calling to nonexisting module $moduleIdentifier and modules controller $bareControllerId");
                throw $e;
            }

            // ziska identifikator specificky pro modul
            $controllerId = Config::getModuleEntryName(
                $bareControllerId,
                $row["identifier"]
            );

            $this->log->info("Complere controllerId = " . $controllerId);

            $routeInfo = new RouteInfo();
            $routeInfo->setUrl($request->getUrl());
            $routeInfo->setPattern($this->adminRoutePattern);
            $routeInfo->setContrlollerId($controllerId);
            $routeInfo->setMethodName($methodName);
            $routeInfo->setInterceptors(array_merge($this->adminInterceptorIdentifiers, $this->adminInternalInterceptorIdentifiers));

            // creates information about called module
            $moduleInfo = $this->createModuleInfo($row, $bareControllerId, $methodName);
            $request->addParameter(ModuleInfo::MODULE_INFO_INDENTIFIER, $moduleInfo);

            // nastavuje id controlleru, ktery se volal
            $request->setCalledControllerId($controllerId);

            return $routeInfo;
        }

        return null;
    }

    /**
     *
     * @param string $controllerId
     * @param array $parameters
     */
    public function findURL($controllerId, array $parameters, $methodName, Locale $locale)
    {
        //        echo $controllerId;

        // login and logout urls
        foreach ($this->adminLoginLogouRoutes as $url => $cId) {
            if ($cId[0] == $controllerId) {
                // TODO: add parameters
                return str_replace("^", "", $url);
            }
        }

        // dashboard and other specific urls
        foreach ($this->adminSpecificRoutes as $url => $cId) {
            if ($cId[0] == $controllerId) {
                return str_replace("^", "", $url);
            }
        }

        if (\IZON\String\startsWith($controllerId, "MODULE#")) { // je cesta k modulu
            $parts = explode("#", $controllerId);
            $typeAndIdentifierPart = explode("|", $parts[1]);

            $url = "/admin/" . $typeAndIdentifierPart[0] . "/" . $parts[2] . "/" . $parameters["action"];
            unset($parameters["action"]);

            $paramsString = "";
            foreach ($parameters as $key => $value) {
                if ($paramsString != '') {
                    $paramsString .= "&";
                }
                $paramsString .= "$key=$value";
            }

            if ($paramsString != '') {
                $url = $url . "?" . $paramsString;
            }
            return $url;
        }

        return null;
    }

    /**
     *
     * @param array $row
     * @param string $controllerId
     * @param string $actionName
     * @return ModuleInfo informace o modulu, na ktery se routuje
     */
    protected function createModuleInfo(
        array $row,
        $controllerId,
        $actionName
    ) {
        $moduleInfo = new ModuleInfo();

        $moduleInfo->setType($row["type"]);
        $moduleInfo->setIdentifier($row["identifier"]);
        $moduleInfo->setName($row["name"]);
        $moduleInfo->setControllerId($controllerId);
        $moduleInfo->setAction($actionName);

        // TODO: nacist nazev akce
        $moduleInfo->setActionName($actionName);

        return $moduleInfo;
    }
}
