<?php

namespace IZON\Admin;

use Exception;
use IZON\Admin\DI\ModulesAwareContainerBuilder;
use IZON\Admin\Domain\AdminModule;
use IZON\Admin\MVC\Exceptions\Handlers\AdminPageNotFoundExceptionHandler;
use IZON\Admin\Services\AdminService;
use IZON\DB\Dao;
use IZON\DB\DaoFactoryBean;
use IZON\Logs\Logger;
use IZON\MVC\Exceptions\Handlers\InternalServerErrorExceptionHandler;
use IZON\MVC\Exceptions\Handlers\PageNotFoundExceptionHandler;
use Psr\Container\ContainerInterface;
use ReflectionClass;
use function IZON\DI\factory;
use function IZON\DI\get;
use function IZON\String\endsWith;

//use \DirectoryIterator;

/**
 * slouzi k nacteni konfigurace i pro admin
 */
class Config extends \IZON\MVC\Config
{
    /**
     * do jakeho souboru se maji zapsat informce o nainstalovanych specificky modulech
     */
    public const INSTALLED_MODULES_FILENAME = "installed-modules.php";

    /**
     * jak se jmenuje soubor obsahujici konfiguraci admina v modulu admina
     *
     */
    public const ADMIN_CONFIG_FILENAME = "admin-config.php";

    /**
     * jak se jmenuje spolecna konfigurace pro modul
     */
    public const MODULE_COMMON_CONFIG_FILENAME = "config.php";

    /**
     * jak se jmenuje soubor obsahujici konfiguraci modulu pro praci v adminu
     */
    public const MODULE_CONFIG_FILENAME = "module-config.php";

    /**
     * identigikator defaultni db
     */
    public const DEFAULT_DB_CONNECTION_IDENTIFIER = 'db.connection';

    /**
     * defaultni servis admina
     * @var string
     */
    public const ADMIN_SERVICE_IDENTIFIER = 'izon.admin.services.adminService';

    /**
     * controller ktery slouzi k prihlaseni k aplikaci
     * @var string
     */
    public const LOGIN_CONTROLLER_IDENTIFIER = 'izon.admin.controller.loginController';

    /**
     * controller ktery slouzi k zobrazeni uvodni obrazovky admina
     * @var string
     */
    public const DASHBOARD_CONTROLLER_IDENTIFIER = 'izon.admin.controller.dashboardController';

    /**
     * controller ktery slouzi k prihlaseni k aplikaci
     * @var string
     */
    public const LOGOUT_CONTROLLER_IDENTIFIER = 'izon.admin.controller.logoutController';

    /**
     * interceptor pro uvereni prihlaseneho uzivatele
     */
    public const LOGGED_USER_CHECK_INTERCEPTOR_IDENTIFIER = 'izon.admin.interceptors.loggedUserCheckInterceptor';

    /**
     * interceptor pro nahrani informaci o prihlasenem uzivateli
     */
    public const LOGGED_USER_INTERCEPTOR_IDENTIFIER = 'izon.admin.interceptors.loggedUserInterceptor';

    /**
     * interceptor pro nacitani akci modulu
     */
    public const MODULES_ACTIONS_INTERCEPTOR_IDENTIFIER = 'izon.admin.interceptors.modulesActionsInterceptor';

    /**
     * interceptor pro nacitani akci modulu
     */
    public const LOAD_ADMIN_VERSION_INTERCEPTOR_IDENTIFIER = 'izon.admin.interceptors.loadAdminVersionInterceptor';

    /**
     * controller ktery slouzi k prihlaseni k aplikaci
     * @var string
     */
    public const NOTIFICATIONS_CONTROLLER_IDENTIFIER = 'izon.admin.controller.notificationsController';

    /**
     * pouziva se pro pedani informace o nacitanych konfiguracich modulu
     * obsahuje typ modulu
     * @var string
     */
    public static $moduleType;

    /**
     * pouziva se pro pedani informace o nacitanych konfiguracich modulu
     * obsahuje unikatni identifikator modulu
     * @var string
     */
    public static $moduleIdentifier;

    /**
     * true pokud prave probiha nacitani modulu
     */
    public static $moduleLoading;

    /**
     * informace o cestach k modulum
     * @var array
     */
    protected static $moduleDirs = null;

    protected static $extConfig;

    /**
     * cesta k souboru se zakladni konfiguraci admina
     * @var string
     */
    protected $adminConfigFileDir = null;

    /**
     * servis pro administraci
     * @var AdminService
     */
    protected $adminService = null;

    /**
     * adresar, kde se nachazeji moduly
     * @var string
     */
    protected $modulesDir = "modules";

    /**
     * obsahuje mista kam se maji dat moduly
     * @var array
     */
    protected $modulesLocations = [];

    /**
     *
     * @var Logger
     */
    protected $log = null;

    public function __construct(
        $baseDir,
        $containerBuilderClass = ModulesAwareContainerBuilder::class
    ) {
        parent::__construct($baseDir, $containerBuilderClass);

        // konfigurace pro servisy a controllery admina
        // TODO: zmenit, aby se bralo nejak inteligentneji
        $this->adminConfigFileDir = dirname(__DIR__) . "/" . self::ADMIN_CONFIG_FILENAME;
    }

    /**
     * vrati hodnotu parametru z konfigu
     *
     * @param string $param Entry name
     * @return string
     */
    public static function getValue($param)
    {
        if (is_array(self::$extConfig)) {
            return self::$extConfig[$param];
        } else {
            return self::$extConfig->get($param);
        }
    }

    /**
     * vrati identifikator hodnoty z modulu
     * @param string $entryName
     * @param string $moduleIdentifier
     */
    public static function getModuleEntryName(
        $entryName,
        $moduleIdentifier
    ) {
        //        echo "Calling getModuleEntryName with parameters: [$entryName, $moduleType, $moduleIdentifier]";

        if ($moduleIdentifier == null
            && !self::$moduleLoading) { // neprovadi se nacitani neceho z modulu v modulu
            $message = "pri odkazovani se na zaznam z modulu je traba zadat \$moduleIdentifier";
            throw new Exception($message);
        }

        if ($moduleIdentifier == null) {
            $moduleIdentifier = self::$moduleIdentifier;
        }

        return self::composeModuleEntryName(
            $entryName,
            $moduleIdentifier
        );
    }

    /**
     * @return array dirs containing modules
     */
    public static function getModulesDirs()
    {
        if (self::$moduleDirs === null) {
            throw new Exception("Modules locations not intialized.");
        }
        return self::$moduleDirs;
    }

    /**
     * TODO: upravit init, aby provdel inicializaci v podtride
     */
    public function init()
    {
        $containerBuilderClass = $this->containerBuilderClass;

        // pole, do ktereho se ukladaji definice
        $definitionsArray = [];
        // nastavuje zakladni cesty v aplikaci
        $pathDefinitions = [
            self::APP_DIR => $this->baseDir,
            self::TMP_DIR_IDENTIFIER => $this->baseDir . "/" . $this->tmpDir,
            self::CACHE_DIR_IDENTIFIER => $this->baseDir . "/" . $this->cacheDir,
        ];
        $definitionsArray = array_merge($definitionsArray, $pathDefinitions);

        // vrati cestu ke konfigu
        $configDir = $this->getAbsoluteConfigDir();

        // nacte zakladni konfiguraci
        $rootConfig = $configDir . "/" . $this->configFilePrefix . ".php";
        if (!file_exists($rootConfig)) {
            $message = "Root config $rootConfig doesn't exist.";
            throw new Exception($message);
        }
        // docasny konfig dokud se nevytvori di kontainer
        $tempConfig = require $rootConfig;

        // nacte specificke konfigurace podle MACHINE_ID
        if (getenv($this->machineIdKey) != '') {
            $machineinfo = explode('-', getenv($this->machineIdKey));

            for ($i = 1; $i <= count($machineinfo); $i++) {
                $tmp = '';
                for ($j = 0; $j < $i; $j++) {
                    $tmp .= '-' . $machineinfo[$j];
                }
                $filePath = $configDir . "/" . $this->configFilePrefix . $tmp . '.php';
                if (file_exists($filePath)) {
                    //                    $this->log->info( "Adding config ". $filePath);
                    $develConfigArray = require $filePath;
                    // slit s korenovou konfiguraci
                    $tempConfig = array_merge($tempConfig, $develConfigArray);
                }
            }
        }

        // pouziti cache pokud je povolena a vytvorena
        $useConfigCache = isset($tempConfig[self::USE_CONFIG_CACHE_IDENTIFIER]) ? $tempConfig[self::USE_CONFIG_CACHE_IDENTIFIER] : false;
        $cacheDir = $this->baseDir . "/" . $this->cacheDir;
        $definitionsCacheFileName = $cacheDir . "/" . $this->configCacheFileName;
        if ($useConfigCache  // ma se pouzit cache definic
            && file_exists($definitionsCacheFileName) // existuje soubor s cache
            && is_file($definitionsCacheFileName)) { // a je to soubor
            // podivej se jestli mas cache a pokud ano tak ji pouzij

            $fileContent = file_get_contents($definitionsCacheFileName);
            $unserialized = unserialize($fileContent);
            $definitions = $unserialized["definitions"];
            self::$moduleDirs = $unserialized["modulesDirs"];

            // nastavuje devel server promennou
            self::$develserver = $definitions["develserver"];

            $cachedBuilder = new $containerBuilderClass();
            $cachedBuilder->addDefinitions($definitions);
            $container = $cachedBuilder->build();

            self::$extConfig = $container;
            // init logger
            $container->get(self::INIT_LOGGER_IDENTIFIER);
            // set logger
            $this->log = Logger::getLogger(self::class);

            $this->container = $container;

            return;
        }

        // nastavuje develserver podle konfigu
        self::$develserver = $tempConfig["develserver"];
        self::$extConfig = $tempConfig;

        // zjisti jestli existuje config admina
        $definitionsArray = array_merge($definitionsArray, $tempConfig);
        if (!file_exists($this->adminConfigFileDir)
            || !is_file($this->adminConfigFileDir)) {
            throw new Exception("Neexistuje soubor " . $this->adminConfigFileDir . " s konfiguraci admina");
        }
        // nacita konfig admina
        $adminConfig = require $this->adminConfigFileDir;
        $definitionsArray = array_merge($definitionsArray, $adminConfig);


        // vytvorit builder pro spravu definic
        $builder = new $containerBuilderClass();
        $builder->addDefinitions($definitionsArray);

        // docasne vytvori container, aby mohl vytahnout servis pro praci s adminem
        $this->container = $builder->build();
        if (!$this->container->has(self::ADMIN_SERVICE_IDENTIFIER)) {
            throw new Exception("Nema servis pro praci s adminem");
        }

        // zpristupnuje config pres volani Config::getConfig()
        self::$extConfig = $this->container;
        $this->container->get(self::INIT_LOGGER_IDENTIFIER);
        // now we can use normal logger
        $this->log = Logger::getLogger(self::class);


        // vytahne si servis pro admina
        $this->adminService = $this->container->get(self::ADMIN_SERVICE_IDENTIFIER);
        // locations of modules
        $this->modulesLocations = require $this->getAbsoluteConfigDir() . "/" . self::INSTALLED_MODULES_FILENAME;

        // vytvari include daa
        // TODO: optimalizovat aby se soubor vkladat jen jednou a vytvorila se cache
        // nacte vsechny daa aplikacce
        $appSrcDir = $this->getBaseDir() . "/app/src";
        foreach (glob($appSrcDir . "/*/*Dao.php") as $file) {
            if (in_array($file, ['.', '..'])) {
                continue;
            }
            require_once $file;
        }
        // nacita daa modulu
        foreach ($this->modulesLocations as $moduleLocation) {
            foreach ($moduleLocation['daoDirs'] as $daoDir) {
                foreach (glob($daoDir . "/*Dao.php") as $file) {
                    if (in_array($file, ['.', '..'])) {
                        continue;
                    }
                    require_once $file;
                }
            }
        }
        // vytvori includovaci array
        $daoInterfacesArray = [];
        //        var_dump(get_declared_interfaces());
        foreach (get_declared_interfaces() as $interface) {
            $reflector = new ReflectionClass($interface);
            if ($reflector->isSubclassOf(Dao::class)) {
                $daoClass = $interface;
                $daoInterfacesArray[$daoClass] = factory(function (ContainerInterface $container) use ($daoClass) {
                    $bean = new DaoFactoryBean($container->get(self::DEFAULT_DB_CONNECTION_IDENTIFIER), $daoClass);
                    return $bean->getObject();
                });
            }
        }

        // vytvorit builder pro spravu definic
        $builder = new $containerBuilderClass();
        $builder->addDefinitions($definitionsArray);

        //        var_dump($daoInterfacesArray);
        $builder->addDefinitions($daoInterfacesArray);
        $definitionsArray = array_merge($definitionsArray, $daoInterfacesArray);

        // nacist konfigurace modulu
        self::$moduleDirs = [];
        $modulesActions = []; // akce modulu co se maji zobrazovat v pravem sloupci
        $modules = $this->adminService->getActiveModules();

        foreach ($modules as $module) {
            self::$moduleType = $module->getType();
            self::$moduleIdentifier = $module->getIdentifier();
            self::$moduleLoading = true;

            $this->log->info("Loading module " . $module->getType() . " with identifier " . $module->getIdentifier());

            if (!isset($this->modulesLocations[$module->getType()])) {
                throw new Exception("Module " . $module->getType() . " not installed.");
            }
            // adresar ve kterem se modul nachazi specificky pro web
            $moduleDir = $this->modulesLocations[$module->getType()]["path"];

            $this->log->info("Trying module dir " . $moduleDir);

            // existuje adresar takoveho modulu
            if (file_exists($moduleDir)
                && is_readable($moduleDir)) {
                // prida konfigurace modulu
                $deff = [];
                $moduleActions = $this->loadAdminModuleConfig($builder, $module, $moduleDir, $deff);
                $modulesActions[$module->getIdentifier()] = [
                    "name" => $module->getName(),
                    "type" => $module->getType(),
                    "identifier" => $module->getIdentifier(),
                    "icon" => $module->getIcon(),
                    "actions" => $moduleActions["actions"],
                    "defaultAction" => $moduleActions["defaultAction"],
                    "menuActions" => $moduleActions["menuActions"],
                ];

                // prida cestu k modulu do cache
                self::$moduleDirs[$module->getType()] = $moduleDir;
            } else {
                $eMsg = "Dir $moduleDir for module " . $module->getType() . " doesn't not exist or isn't writable.";
                $this->log->error($eMsg);
                throw new Exception($eMsg);
            }

            $definitionsArray = array_merge($definitionsArray, $deff);
            self::$moduleLoading = false;
        }

        // TODO: brat identifier nekce z neceho inteligentnejsiho
        $modulesActionsDefinition = ["admin.modulesActions" => $modulesActions];
        $builder->addDefinitions($modulesActionsDefinition);
        $definitionsArray = array_merge($definitionsArray, $modulesActionsDefinition);

        // nacist ostatni konfigurace webu
        $webDir = $this->getAbsoluteConfigDir() . "/web";
        foreach (scandir($webDir) as $file) {
            if (endsWith($file, ".php")) {
                $controllerDefinitions = require $webDir . "/" . $file;
                $builder->addDefinitions($controllerDefinitions);
                $definitionsArray = array_merge($definitionsArray, $controllerDefinitions);
            }
        }

        // vytbori di container
        $this->container = $builder->build();

        // vytvor cache pokud je to povoleno
        if ($this->container->has(self::USE_CONFIG_CACHE_IDENTIFIER)
            && $this->container->get(self::USE_CONFIG_CACHE_IDENTIFIER)) { // ma se pouzit cache definic
            // vytvorit cache definic
            $fileContent = ["definitions" => $definitionsArray, "modulesDirs" => self::$moduleDirs];
            $serialized = serialize($fileContent);
            if (!file_exists($cacheDir)) { // vytvorit cache dir
                mkdir($cacheDir, 0777, true);
            }
            // zapis data do souboru
            file_put_contents($definitionsCacheFileName, $serialized, LOCK_EX);
        }
    }

    /**
     * nacte konfiguraci pro admina daneho modulu
     * @param ModulesAwareContainerBuilder $builder builder pro di
     * @param string $moduleDir v jakem adresari se modul nachazi
     */
    protected function loadAdminModuleConfig(
        ModulesAwareContainerBuilder $builder,
        AdminModule $module,
        $moduleDir,
        &$definitionsArray
    ) {
        // nacist zakladni konfiguraci modulu
        $configFileDir = $moduleDir . "/" . self::MODULE_COMMON_CONFIG_FILENAME;
        if (file_exists($configFileDir)) { // TODO: resit ze modul nema konfig
            $moduleConfArray = include $configFileDir;

            $newModuleConfArray = [];

            foreach ($moduleConfArray as $key => $value) {
                $propertyNewName = self::composeModuleEntryName(
                    $key,
                    $module->getIdentifier()
                );
                $newModuleConfArray[$propertyNewName] = $value;

                $this->log->info("Addnig config $propertyNewName to builder");
            }
            $builder->addDefinitions($newModuleConfArray);
            $definitionsArray = array_merge($definitionsArray, $newModuleConfArray);
        }

        // nacist konfiguraci pro admina
        $configFileDir = $moduleDir . "/" . self::MODULE_CONFIG_FILENAME;
        if (file_exists($configFileDir)) { // TODO: resit ze modul nema konfig
            $moduleConfArray = include $configFileDir;

            $newModuleConfArray = [];

            // vytvori identifikatory nezavisle pro jednotlive nactene moduly
            foreach ($moduleConfArray as $key => $value) {
                $propertyNewName = self::composeModuleEntryName(
                    $key,
                    $module->getIdentifier()
                );

                $newModuleConfArray[$propertyNewName] = $value;

                $this->log->info("Addnig config $propertyNewName to builder");
            }
            $builder->addDefinitions($newModuleConfArray);
            $definitionsArray = array_merge($definitionsArray, $newModuleConfArray);

            // nacist akce modulu
            if (!isset($moduleConfArray["moduleActions"])) {
                throw new Exception(
                    "Module " . $module->getType() . " doesn't have actions definition [array key: moduleActions]"
                );
            }
            return $moduleConfArray["moduleActions"];
        }
    }

    /**
     * slozi identifikator pro zaznam v konfiguraci v adminu unikatni pro nacitani jednotlivych modulu
     * @param string $entryName zaznam v konfiguraci modulu
     * @param string $moduleIdentifier identifikator
     * @return string konecny identifikator pod ktery se ulozi do DI kontejneru
     */
    protected static function composeModuleEntryName(
        $entryName,
        $moduleIdentifier
    ) {
        return "MODULE#$moduleIdentifier#$entryName";
    }

    /**
     * returns exception handlers for aplication
     */
    public function getExceptionHandlers()
    {
        $container = $this->container;
        if ($container->has(self::EXCEPTION_HANDLERS_INDENTIFIER)) {
            $handlers = $container->get(self::EXCEPTION_HANDLERS_INDENTIFIER);
            $this->log->info("Using config exception handlers");
        } else {
            // use default
            $handlers = [];
            $handlers[] = new PageNotFoundExceptionHandler();
            $handlers[] = new AdminPageNotFoundExceptionHandler();
            $handlers[] = new InternalServerErrorExceptionHandler();

            $this->log->info("Using default exception handlers");
        }

        return $handlers;
    }
}
