<?php

namespace IZON\Modules\AdminUsers\Admin\Services;

use IZON\Admin\Dao\AdminRoleDao;
use IZON\Admin\Dao\AdminUserDao;
use IZON\Admin\Dao\AdminUserRoleDao;
use IZON\Admin\Domain\AdminUser;
use IZON\Admin\Domain\AdminUserRole;
use IZON\Admin\Services\AbstractBaseService;
use IZON\Admin\Services\LoggedAdminUserSessionServiceInterface;
use IZON\DB\OrderBy;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\DBLocale\Dao\DBCountryDao;
use IZON\DBLocale\Dao\DBLanguageDao;
use IZON\DBLocale\Dao\DBLocaleDao;

/**
 * servis pro zpracovani referenci
 */
class UserService extends AbstractBaseService
{
    /**
     * @var LoggedAdminUserSessionServiceInterface
     */
    protected $sessionService;


    /**
     * @var AdminUserDao
     */
    protected $userDao;

    /**
     * @var AdminRoleDao
     */
    protected $adminRoleDao;

    /**
     * @var AdminUserRoleDao
     */
    protected $adminUserRoleDao;

    /**
     * @var DBCountryDao
     */
    protected $countryDao;

    /**
     * @var DBLanguageDao
     */
    protected $languageDao;

    /**
     * @var DBLocaleDao
     */
    protected $localeDao;


    /// generovany konstruktor
    public function __construct(LoggedAdminUserSessionServiceInterface $sessionService, AdminUserDao $userDao, AdminRoleDao $adminRoleDao, AdminUserRoleDao $adminUserRoleDao, DBCountryDao $countryDao, DBLanguageDao $languageDao, DBLocaleDao $localeDao)
    {
        $this->sessionService = $sessionService;
        $this->userDao = $userDao;
        $this->adminRoleDao = $adminRoleDao;
        $this->adminUserRoleDao = $adminUserRoleDao;
        $this->countryDao = $countryDao;
        $this->languageDao = $languageDao;
        $this->localeDao = $localeDao;
        $this->dao = $userDao;
    }


    /**
     * vrati defaultni strankovac
     */
    public function getPaginatorConfig($maxPageSize)
    {
        return $this->userDao->createDefaultPaginatorConfig($maxPageSize);
    }

    /**
     * vrati uživatele
     */
    public function getPage(PaginatorConfig $config)
    {
        $users = $this->userDao->paginate($config);
        foreach ($users->getPage() as $user) {
            $this->fillAdminUser($user);
        }
        return $users;
    }

    /**
     * @param integer $id
     * @return AdminUser
     */
    public function get($id)
    {
        $user = $this->userDao->load($id);
        $this->fillAdminUser($user);

        return $user;
    }

    public function update(AdminUser $obj)
    {
        $this->userDao->update($obj);
        $this->saveUserRoles($obj);
    }

    public function save(AdminUser $obj)
    {
        $id = $this->userDao->save($obj);
        $this->saveUserRoles($obj);
        return $id;
    }

    public function delete($id)
    {
        $this->userDao->delete($id);
        $oldUserRoles = $this->adminUserRoleDao->find(["fkAdminUserId" => $id])->listResult();
        foreach ($oldUserRoles as $oldUserRole) {
            $this->adminUserRoleDao->delete($oldUserRole);
        }
    }


    public function getRoles()
    {
        return $this->adminRoleDao->find([], ["name" => OrderBy::ASC])->listResult();
    }

    public function getRoleById($roleId)
    {
        return $this->adminRoleDao->load($roleId);
    }

    public function getActiveCountries()
    {
        return $this->countryDao->find(["enabled" => true], ["name" => OrderBy::ASC])->listResult();
    }

    public function getActiveLanguages()
    {
        return $this->languageDao->find(["enabled" => true], ["name" => OrderBy::ASC])->listResult();
    }

    public function getActiveLocales()
    {
        return $this->localeDao->find([], ["localeTag" => OrderBy::ASC])->listResult();
    }

    public function computePasswordHash($password)
    {
        return $this->sessionService->computePasswordHash($password);
    }


    protected function fillAdminUser(AdminUser $adminUser)
    {
        $roles = $this->adminRoleDao->findUserRoles($adminUser->getId())->listResult();
        $adminUser->setRoles($roles);
    }

    protected function saveUserRoles(AdminUser $adminUser)
    {
        $oldUserRoles = $this->adminUserRoleDao->find(["fkAdminUserId" => $adminUser->getId()])->listResult();
        $oldUserRolesHash = [];
        foreach ($oldUserRoles as $oldUserRole) {
            $oldUserRolesHash[$oldUserRole->getFkAdminRoleId()] = $oldUserRole;
        }

        foreach ($adminUser->getRoles() as $role) {
            if (isset($oldUserRolesHash[$role->getId()])) {
                unset($oldUserRolesHash[$role->getId()]);
            } else {
                $userRole = new AdminUserRole();
                $userRole->setFkAdminUserId($adminUser->getId());
                $userRole->setFkAdminRoleId($role->getId());

                $this->adminRoleDao->save($userRole);
            }
        }

        foreach ($oldUserRolesHash as $oldUserRole) {
            $this->adminRoleDao->delete($oldUserRole);
        }
    }
}
