<?php

namespace IZON\Modules\AdminUsers\Admin\Controllers;

use Exception;
use IZON\Admin\Domain\AdminRole;
use IZON\Admin\Domain\AdminUser;
use IZON\Admin\Services\LoggedAdminUserSessionServiceInterface;
use IZON\Admin\Web\Controllers\AbstractBaseController;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\Forms\Form;
use IZON\Modules\AdminUsers\Admin\Forms\AdminUserFormFactory;
use IZON\Modules\AdminUsers\Admin\Services\UserService;
use IZON\MVC\HttpRequest;
use IZON\MVC\ModelAndView;

/**
 * @extends AbstractBaseController<UserService>
 */
class UserController extends AbstractBaseController
{
    /** @var LoggedAdminUserSessionServiceInterface  */
    protected $sessionService;

    /** @var UserService  */
    protected $userService;

    /** @var int */
    protected $maxPageSize = 1000;

    /** @var string  */
    protected $rolesPreffix = "roles";

    public function __construct(LoggedAdminUserSessionServiceInterface $sessionService, UserService $userService, $viewName)
    {
        parent::__construct($userService);

        $this->sessionService = $sessionService;
        $this->userService = $userService;
        $this->viewName = $viewName;
    }

    /**
     * execute je defaultni akce, ktera se zavola, pokud neni specifikovana jina metoda
     * @param HttpRequest $request
     * @return ModelAndView
     */
    public function execute(HttpRequest $request)
    {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/list");

        $paginatorConfig = $this->userService->getPaginatorConfig($this->maxPageSize);

        $modelAndView->putParameter("paginatorConfig", $paginatorConfig);

        $modelAndView->putParameter("pageContent", $this->userService->getPage($paginatorConfig));

        return $modelAndView;
    }

    /**
     * zobrazi editacni formular a naplni ho daty
     */
    public function edit(HttpRequest $request)
    {
        if ($request->hasParameter("id")) { // posila se id, musi se editovat odpovidajici user
            $user = $this->userService->get($request->getParameter("id"));

            $loggedUser = $this->sessionService->getLoggedUser();

            if ($user->getSuperuser() && !$loggedUser->getSuperuser()) {
                return new ModelAndView($request->getCalledControllerId(), ['action' => 'execute'], true);
            }
        } else {
            $user = $this->createDomainObject();
        }

        $activeCountries = $this->userService->getActiveCountries();
        $activeLanguages = $this->userService->getActiveLanguages();
        $activeLocales = $this->userService->getActiveLocales();

        $rolesFormsHash = [];
        $roles = [];
        /** @var AdminRole $role */
        foreach ($this->userService->getRoles() as $role) {
            $roles[$role->getId()] = $role;
            $roleForm = $this->createRoleForm([$this->rolesPreffix, $role->getId()], $role->getName());
            $rolesFormsHash[$role->getId()] = $roleForm;
        }

        $form = $this->createForm($activeCountries, $activeLanguages, $activeLocales);
        $form->edit($user);

        foreach ($user->getRoles() as $role) {
            $formRole = $rolesFormsHash[$role->getId()];
            $formRole->getField("hasRole")->serialize(true);
        }

        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/form");
        $modelAndView->putParameter("form", $form);
        $modelAndView->putParameter("roles", $roles);
        $modelAndView->putParameter("roleForms", $rolesFormsHash);

        return $modelAndView;
    }

    public function save(HttpRequest $request)
    {
        $activeCountries = $this->userService->getActiveCountries();
        $activeLanguages = $this->userService->getActiveLanguages();
        $activeLocales = $this->userService->getActiveLocales();

        $form = $this->createForm($activeCountries, $activeLanguages, $activeLocales);
        $form->setValues($request->getParameters());

        $roles = $this->userService->getRoles();
        $selectedRoles = [];
        foreach ($roles as $role) {
            $roleForm = $this->createRoleForm([$this->rolesPreffix, $role->getId()], $role->getName());
            $roleForm->setValues($request->getParameters());

            if ($roleForm->getField("hasRole")->parse()) {
                $selectedRoles[] = $role;
            }
        }

        if ($request->hasParameter("id")
            && $request->getParameter("id")
            != '') { // posila se id, musi se editovat odpovidajici objekt
            $id = $request->getParameter("id");

            $object = $this->userService->get($id);
            $oldPasswordHash = $object->getPassword();

            $form->flush($object);

            if ($object->getPassword() != '') { // men heslo jen pokud je vyplnene
                $object->setPassword($this->userService->computePasswordHash($object->getPassword()));
            } else {
                $object->setPassword($oldPasswordHash);
            }
            $object->setRoles($selectedRoles);

            $this->userService->update($object);
        } else {
            // TODO: potom odstranit
            $object = $this->createDomainObject();
            $form->flush($object);
            $object->setRoles($selectedRoles);

            // vytvor hash vybraneho hesla
            $object->setPassword($this->userService->computePasswordHash($object->getPassword()));

            $this->userService->save($object);
        }

        // TODO: predavat nejak informaci o this module nebo tak neco a nepouzivat Config
        $modelAndView = new ModelAndView(
            $request->getCalledControllerId(),
            ['action' => 'execute'],
            true
        );

        return $modelAndView;
    }

    public function delete(HttpRequest $request)
    {
        if (!$request->hasParameter("id")) {
            throw new Exception("Can't delete. No id provided");
        }

        $id = $request->getParameter("id");
        $this->userService->delete($id);

        // TODO: provide flash message

        $modelAndView = new ModelAndView(
            $request->getCalledControllerId(),
            ['action' => 'execute'],
            true
        );

        return $modelAndView;
    }

    protected function createForm($activeCountries, $activeLanguages, $activeLocales)
    {
        return AdminUserFormFactory::createUserForm($activeCountries, $activeLanguages, $activeLocales);
    }

    protected function createRoleForm($preffix, $name)
    {
        return AdminUserFormFactory::createUserRoleForm($preffix, $name);
    }

    protected function createDomainObject()
    {
        $user = new AdminUser();

        $user->setId(null);

        return $user;
    }

    protected function getPage(PaginatorConfig $config)
    {
        // TODO: Implement getPage() method.
    }

    protected function modifyPaginator(PaginatorConfig $paginatorConfig, Form $filterForm)
    {
        // TODO: Implement modifyPaginator() method.
    }

    protected function getPaginatorSessionIndentifier()
    {
        // TODO: Implement getPaginatorSessionIndentifier() method.
    }

    protected function createPaginatorConfig()
    {
        // TODO: Implement createPaginatorConfig() method.
    }

    protected function createFilterForm()
    {
        // TODO: Implement createFilterForm() method.
    }
}
