<?php
namespace IZON\Modules\AdminUsers\Admin\Services;

use IZON\DB\Paginator\PaginatorConfig;
use IZON\DB\OrderBy;

use IZON\Admin\Services\SessionService;

use IZON\DBLocale\Dao\CountryDao;
use IZON\DBLocale\Dao\LanguageDao;
use IZON\DBLocale\Dao\LocaleDao;

use IZON\Admin\Dao\AdminUserDao;
use IZON\Admin\Dao\AdminRoleDao;
use IZON\Admin\Dao\AdminUserRoleDao;

use IZON\Admin\Domain\AdminUser;
use IZON\Admin\Domain\AdminUserRole;

/**
 * servis pro zpracovani referenci
 */
class UserService {    
    
    /**
     * @var SessionService
     */
    protected $sessionService;


    /**
     * @var AdminUserDao 
     */
    protected $userDao;

    /**
     * @var AdminRoleDao 
     */
    protected $adminRoleDao;

    /**
     * @var AdminUserRoleDao
     */
    protected $adminUserRoleDao;

    /**
     * @var CountryDao 
     */
    protected $countryDao;
    
    /**
     * @var LanguageDao 
     */
    protected $languageDao;
    
    /**
     * @var LocaleDao 
     */
    protected $localeDao;
    

/// generovany konstruktor    
    function __construct(SessionService $sessionService, AdminUserDao $userDao, AdminRoleDao $adminRoleDao, AdminUserRoleDao $adminUserRoleDao, CountryDao $countryDao, LanguageDao $languageDao, LocaleDao $localeDao) {
        $this->sessionService = $sessionService;
        $this->userDao = $userDao;
        $this->adminRoleDao = $adminRoleDao;
        $this->adminUserRoleDao = $adminUserRoleDao;
        $this->countryDao = $countryDao;
        $this->languageDao = $languageDao;
        $this->localeDao = $localeDao;
    }

    
    /**
     * vrati defaultni strankovac
     */
    public function getPaginatorConfig($maxPageSize) {
        return $this->userDao->createDefaultPaginatorConfig($maxPageSize);
    }
    
    /**
     * vrati aktivni moduly
     */
    public function getPage(PaginatorConfig $config) {
        return $this->userDao->paginate($config);
    }
    
    /**
     * @param integer $id
     * @return AdminUser
     */
    public function get($id) {
        $user = $this->userDao->load($id);
        $this->fillAdminUser($user);
        
        return $user;
    }
    
    public function update(AdminUser $obj) {
        $this->userDao->update($obj);
        $this->saveUserRoles($obj);
    }
    
    public function save(AdminUser $obj) {
        $id = $this->userDao->save($obj);
        $this->saveUserRoles($obj);
        return $id;
    }

    public function delete($id) {
       $this->userDao->delete($id);
       $oldUserRoles = $this->adminUserRoleDao->find(["fkAdminUserId" => $id])->listResult();
       foreach($oldUserRoles as $oldUserRole) {
            $this->adminUserRoleDao->delete($oldUserRole);
        }
    }
    
    
    public function getRoles() {
        return $this->adminRoleDao->find([], ["name" => OrderBy::ASC])->listResult();
    }
    
    public function getRoleById($roleId) {
      return $this->adminRoleDao->load($roleId);
    }

    public function getActiveCountries() {
        return $this->countryDao->find(["enabled" => true], ["name" => OrderBy::ASC])->listResult();
    }
    
    public function getActiveLanguages() {
        return $this->languageDao->find(["enabled" => true], ["name" => OrderBy::ASC])->listResult();
    }
    
    public function getActiveLocales() {
        return $this->localeDao->find([], ["localeTag" => OrderBy::ASC])->listResult();
    }
    
    public function computePasswordHash($password) {
        return $this->sessionService->computePasswordHash($password);
    }


    protected function fillAdminUser(AdminUser $adminUser) {
        $roles = $this->adminRoleDao->findUserRoles($adminUser->getId())->listResult();
        $adminUser->setRoles($roles);
    }
    
    protected function saveUserRoles(AdminUser $adminUser) {
        $oldUserRoles = $this->adminUserRoleDao->find(["fkAdminUserId" => $adminUser->getId()])->listResult();
        $oldUserRolesHash = [];
        foreach($oldUserRoles as $oldUserRole) {
            $oldUserRolesHash[$oldUserRole->getFkAdminRoleId()] = $oldUserRole;
        }
        
        foreach($adminUser->getRoles() as $role) {
            if( isset($oldUserRolesHash[$role->getId()]) ) {
                unset($oldUserRolesHash[$role->getId()]);
            } else {
                $userRole = new AdminUserRole();
                $userRole->setFkAdminUserId($adminUser->getId());
                $userRole->setFkAdminRoleId($role->getId());
                
                $this->adminRoleDao->save($userRole);
            }
        }

        foreach($oldUserRolesHash as $oldUserRole) {
            $this->adminRoleDao->delete($oldUserRole);
        }
    }
}