<?php

namespace IZON\Modules\AdminModules\Admin\Services;

use Doctrine\Common\Persistence\ObjectRepository;
use Exception;
use IZON\Admin\Config;
use IZON\Admin\Domain\AdminModule;
use IZON\Admin\Domain\AdminModuleSetting;
use IZON\Admin\Domain\AdminRole;
use IZON\Admin\Services\AbstractBaseService;
use IZON\DB\EntityManagerInterface;
use IZON\DB\Paginator\PageContent;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\DB\QueryParams\OrderBy;
use IZON\DB\Repository\BaseRepository;
use IZON\DBLocale\Domain\DBCountry;
use IZON\DBLocale\Domain\DBLanguage;
use IZON\DBLocale\Domain\DBLocale;

/**
 * servis pro zpracovani referenci
 */
class ModuleService extends AbstractBaseService
{
    /**
     * @var string ve kterem adresari se aplikace nachazi
     */
    protected $appDir;

    /**
     * @var string v jakem adresari se nachazeji konfigy aplikace
     */
    protected $configDir;

    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;

    /**
     *
     * @var BaseRepository
     */
    protected $adminModuleRepository;

    /**
     * @var ObjectRepository
     */
    protected $adminModuleSettingRepository;
    /**
     * @var ObjectRepository
     */
    protected $adminRoleRepository;
    /**
     * @var ObjectRepository
     */
    protected $countryRepository;
    /**
     * @var ObjectRepository
     */
    protected $languageRepository;
    /**
     * @var ObjectRepository
     */
    protected $localeRepository;

    /**
     *
     * @param string $appDir
     * @param string $configDir
     * @param EntityManagerInterface $entityManager
     */
    public function __construct($appDir, $configDir, EntityManagerInterface $entityManager)
    {
        $this->appDir = $appDir;
        $this->configDir = $configDir;

        $this->entityManager = $entityManager;
        $this->adminModuleRepository = $entityManager->getRepository(AdminModule::class);
        $this->adminModuleSettingRepository = $entityManager->getRepository(AdminModuleSetting::class);
        $this->adminRoleRepository = $entityManager->getRepository(AdminRole::class);
        $this->countryRepository = $entityManager->getRepository(DBCountry::class);
        $this->languageRepository = $entityManager->getRepository(DBLanguage::class);
        $this->localeRepository = $entityManager->getRepository(DBLocale::class);
    }

    /**
     *
     * @param int $pageSize
     * @return PaginatorConfig
     */
    public function getPaginatorConfig($pageSize)
    {
        $config = $this->adminModuleRepository->createDefaultPaginatorConfig($pageSize);
        $config->addOrder("moduleOrder", OrderBy::ORDER_ASC);
        return $config;
    }

    /**
     *
     * @param PaginatorConfig $config
     * @return PageContent
     */
    public function getPage(PaginatorConfig $config)
    {
        $pageContent = $this->adminModuleRepository->paginate($config);
        return $pageContent;
    }

    /**
     * @param int $id
     * @return AdminModule
     */
    public function get($id)
    {
        $module = $this->adminModuleRepository->find($id);
        return $module;
    }

    /**
     *
     * @param AdminModule $obj
     */
    public function update(AdminModule $obj)
    {
        $this->entityManager->persist($obj);
        $this->entityManager->flush();
    }

    /**
     *
     * @param int $id
     */
    public function delete($id)
    {
        /** @var AdminModule $module */
        $module = $this->adminModuleRepository->find($id);
        $this->entityManager->remove($module);
        $this->entityManager->flush();
    }

    /**
     *
     * @param AdminModule $obj
     */
    public function save(AdminModule $obj)
    {
        $this->entityManager->persist($obj);
        $this->entityManager->flush();
    }

    /**
     *
     * @param AdminModule $module
     */
    public function install(AdminModule $module)
    {
        $this->fillDefaultModuleSettings($module);
        $this->save($module);
    }

    /**
     * @return array vrati typy vsech dostupnych modulu
     */
    public function getAvailableModulesTypes(): array
    {
        $moduleTypes = [];

        $installedModules = require $this->configDir . "/" . Config::INSTALLED_MODULES_FILENAME;
        foreach ($installedModules as $type => $value) {
            if (isset($value["path"])) { // ma cestu k modulu
                $moduleTypes[$type] = $type;
            }
        }
        asort($moduleTypes);

        return $moduleTypes;
    }

    /**
     *
     * @param AdminModule $module
     * @throws Exception
     */
    public function fillDefaultModuleSettings(AdminModule $module)
    {
        $moduleTypes = [];

        $installedModules = require $this->configDir . "/" . Config::INSTALLED_MODULES_FILENAME;

        if (!isset($installedModules[$module->getType()])) { // takovy module neni dostupny
            throw new Exception("Modul/Knihovna " . $module->getType() . " neni definovany v composer.json");
        }
        $moduleConfig = $installedModules[$module->getType()];
        if (!isset($moduleConfig["path"])) {
            throw new Exception($module->getType() . " není modul");
        }

        // sobir s nastavenimi
        $moduleSettingFile = $this->appDir . '/www/' . $moduleConfig["path"] . "/install/settings.php";
        if (file_exists($moduleSettingFile)) { // nastaveni modulu existuje
            $settings = require_once $moduleSettingFile;

            // kopiruje nastaveni
            $moduleSettings = [];
            $module->setIcon($settings["icon"]);
            foreach ($settings["moduleSettings"] as $paramIdentifier => $setting) {
                $adminSetting = new AdminModuleSetting();
                $adminSetting->setParamIdentifier($paramIdentifier);
                $adminSetting->setName($setting["name"]);
                $adminSetting->setValue($setting["value"]);

                $moduleSettings[] = $adminSetting;
            }
            $module->setModuleSettings($moduleSettings);
        }
    }

    /**
     *
     * @return AdminRole[]
     */
    public function getRoles(): array
    {
        return $this->adminRoleRepository->findBy([], ['name' => 'ASC']);
    }

    /**
     *
     * @return DBCountry[]
     */
    public function getActiveCountries(): array
    {
        return $this->countryRepository->findBy([], ["name" => 'ASC']);
    }

    /**
     *
     * @return DBLanguage[]
     */
    public function getActiveLanguages(): array
    {
        //DOTO - bude DBLanguage mít property "enabled" ?
        return $this->languageRepository->findBy([], ["name" => 'ASC']);
    }

    /**
     *
     * @return DBLocale[]
     */
    public function getActiveLocales(): array
    {
        return $this->localeRepository->findBy([], ["localeTag" => 'ASC']);
    }
}
