<?php

namespace IZON\Modules\AdminModules\Admin\Controllers;

use Exception;
use IZON\Admin\Domain\AdminModule;
use IZON\Admin\Domain\AdminRole;
use IZON\Admin\Web\Controllers\AbstractBaseController;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\Forms\Form;
use IZON\Modules\AdminModules\Admin\Forms\AdminModuleFormFactory;
use IZON\Modules\AdminModules\Admin\Services\ModuleService;
use IZON\MVC\HttpRequest;
use IZON\MVC\ModelAndView;

class ModuleController extends AbstractBaseController {
    /**
     * @var ModuleService
     */
    protected $moduleService;
    /**
     *
     * @var string name of view to call
     */
    protected $viewName;
    protected $maxPageSize = 1000;
    protected $rolesPreffix = "roles";
    protected $settingsPreffix = "settings";

    public function __construct(ModuleService $moduleService, $viewName) {
        parent::__construct($moduleService);
        $this->moduleService = $moduleService;
        $this->viewName = $viewName;
    }

    /**
     * execute je defaultni akce, ktera se zavola, pokud neni specifikovana jina metoda
     * @param HttpRequest $request
     * @return ModelAndView
     */
    public function execute(HttpRequest $request) {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/list");

        $paginatorConfig = $this->moduleService->getPaginatorConfig($this->maxPageSize);
        $paginatorConfig->clearOrders();
        $paginatorConfig->addOrder("moduleOrder");

        $modelAndView->putParameter("paginatorConfig", $paginatorConfig);

        $modelAndView->putParameter("pageContent", $this->moduleService->getPage($paginatorConfig));

        return $modelAndView;
    }

    /**
     * zobrazi editacni formular a naplni ho daty
     */
    public function edit(HttpRequest $request) {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/form");

        $activeCountries = $this->moduleService->getActiveCountries();
        $activeLanguages = $this->moduleService->getActiveLanguages();
        $activeLocales = $this->moduleService->getActiveLocales();

        $form = $this->createForm($activeCountries, $activeLanguages, $activeLocales);
        $modelAndView->putParameter("form", $form);

        $rolesFormsHash = [];
        $roles = $this->moduleService->getRoles();
        foreach($roles as $role) {
            $roleForm = $this->createRoleForm([$this->rolesPreffix, $role->getId()], $role->getName());
            $rolesFormsHash[$role->getId()] = $roleForm;
        }
        $modelAndView->putParameter("roleForms", $rolesFormsHash);

        if($request->hasParameter("id")) { // posila se id, musi se editovat odpovidajici objekt
            $id = $request->getParameter("id");

            $object = $this->moduleService->get($id);
            $form->edit($object);

            foreach($object->getRoles() as $role) {
                /* @var $role AdminRole */
                $form = $rolesFormsHash[$role->getId()];
                $form->getField("hasRole")->serialize(true);
            }

            $moduleSettingForms = [];
            foreach($object->getModuleSettings() as $moduleSetting) {
                $settingForm = $this->createModuleSettingsForm([$this->settingsPreffix, $moduleSetting->getId()]);
                $settingForm->edit($moduleSetting);

                $moduleSettingForms[] = $settingForm;
            }
            $modelAndView->putParameter("settingForms", $moduleSettingForms);
        } else {
            $article = $this->createDomainObject();
            $form->edit($article);
        }

        return $modelAndView;
    }

    public function save(HttpRequest $request) {
        $activeCountries = $this->moduleService->getActiveCountries();
        $activeLanguages = $this->moduleService->getActiveLanguages();
        $activeLocales = $this->moduleService->getActiveLocales();

        $form = $this->createForm($activeCountries, $activeLanguages, $activeLocales);
        $form->setValues($request->getParameters());

        $roles = $this->moduleService->getRoles();
        $selectedRoles = [];
        foreach($roles as $role) {
            $roleForm = $this->createRoleForm([$this->rolesPreffix, $role->getId()], $role->getName());
            $roleForm->setValues($request->getParameters());

            if($roleForm->getField("hasRole")->parse()) {
                $selectedRoles[] = $role;
            }
        }

        if($request->hasParameter("id") && $request->getParameter("id") != '') { // posila se id, musi se editovat odpovidajici objekt
            $id = $request->getParameter("id");

            $object = $this->moduleService->get($id);
            $form->flush($object);
            $object->setRoles($selectedRoles);

            foreach($object->getModuleSettings() as $moduleSetting) {
                $settingForm = $this->createModuleSettingsForm([$this->settingsPreffix, $moduleSetting->getId()]);
                $settingForm->setValues($request->getParameters());
                $settingForm->flush($moduleSetting);
            }

            $this->moduleService->update($object);
        } else {
            throw new Exception("Modul se neda vytvorit timto zpusobem.");
        }

        // TODO: predavat nejak informaci o this module nebo tak neco a nepouzivat Config
        $modelAndView = new ModelAndView(
            $request->getCalledControllerId(), ['action' => 'execute'], true);

        return $modelAndView;
    }

    /**
     * zobrazi formular por instalaci noveho modulu
     */
    public function installForm(HttpRequest $request) {
        $modelAndView = new ModelAndView("admin/" . $this->viewName . "/install-form");

        $availableModulesTypes = $this->moduleService->getAvailableModulesTypes();

        $form = $this->createModuleInstallForm($availableModulesTypes);
        $modelAndView->putParameter("form", $form);

        return $modelAndView;
    }

    public function install(HttpRequest $request) {
        $availableModulesTypes = $this->moduleService->getAvailableModulesTypes();

        $form = $this->createModuleInstallForm($availableModulesTypes);
        $form->setValues($request->getParameters());

        $form->validate();
        if($form->hasErrors()) {
            $modelAndView = new ModelAndView("admin/" . $this->viewName . "/install-form");
            return $modelAndView;
        }

        $module = $this->createDomainObject();
        $form->flush($module);

        $this->moduleService->install($module);

        $modelAndView = new ModelAndView(
            $request->getCalledControllerId(), ['action' => 'edit', "id" => $module->getId()], true);

        return $modelAndView;
    }

    public function delete(HttpRequest $request) {
        if(!$request->hasParameter("id")) {
            throw new Exception("Can't delete. No id provided");
        }

        $id = $request->getParameter("id");
        $this->moduleService->delete($id);

        // TODO: provide flash message

        $modelAndView = new ModelAndView(
            $request->getCalledControllerId(), ['action' => 'execute'], true);

        return $modelAndView;
    }

    protected function createForm($activeCountries, $activeLanguages, $activeLocales) {
        return AdminModuleFormFactory::createModuleForm($activeCountries, $activeLanguages, $activeLocales);
    }

    protected function createRoleForm($preffix, $name) {
        return AdminModuleFormFactory::createRoleModuleForm($preffix, $name);
    }

    protected function createModuleSettingsForm($preffix) {
        return AdminModuleFormFactory::createModuleSettingForm($preffix);
    }

    protected function createModuleInstallForm($modulesSetings) {
        return AdminModuleFormFactory::createModuleInstallForm($modulesSetings);
    }

    protected function createDomainObject() {
        $user = new AdminModule();

        $user->setId(NULL);
        $user->setEnabled(false);
        $user->setAdminVisible(false);
        $user->setModuleOrder(0);
        $user->setIcon('');

        return $user;
    }

    protected function getPage(PaginatorConfig $config) {
        // TODO: Implement getPage() method.
    }

    protected function modifyPaginator(PaginatorConfig $paginatorConfig, Form $filterForm) {
        // TODO: Implement modifyPaginator() method.
    }

    protected function getPaginatorSessionIndentifier() {
        // TODO: Implement getPaginatorSessionIndentifier() method.
    }

    protected function createPaginatorConfig() {
        // TODO: Implement createPaginatorConfig() method.
    }

    protected function createFilterForm() {
        // TODO: Implement createFilterForm() method.
    }


}
