<?php
namespace IZON\AdminModules\Admin\Services;

use Exception;

use IZON\Admin\Config;

use IZON\DB\Paginator\PaginatorConfig;
use IZON\DB\OrderBy;


use IZON\DBLocale\Dao\CountryDao;
use IZON\DBLocale\Dao\LanguageDao;
use IZON\DBLocale\Dao\LocaleDao;

use IZON\Admin\Dao\AdminModuleDao;
use IZON\Admin\Dao\AdminModuleSettingDao;
use IZON\Admin\Dao\AdminRoleDao;
use IZON\Admin\Dao\AdminRoleModuleDao;

use IZON\Admin\Domain\AdminModule;
use IZON\Admin\Domain\AdminModuleSetting;
use IZON\Admin\Domain\AdminRoleModule;

/**
 * servis pro zpracovani referenci
 */
class ModuleService {
    
    /**
     * @var string ve kterem adresari se aplikace nachazi 
     */
    protected $appDir;

    /**
     * @var string v jakem adresari se nachazeji konfigy aplikace 
     */
    protected $configDir;


    /**
     * @var AdminModuleDao 
     */
    protected $adminModuleDao;

    /**
     * @var AdminModuleSettingDao 
     */
    protected $adminModuleSettingDao;

    /**
     * @var AdminRoleDao
     */
    protected $adminRoleDao;
    
    /**
     * @var AdminRoleModuleDao
     */
    protected $adminRoleModuleDao;

    /**
     * @var CountryDao 
     */
    protected $countryDao;
    
    /**
     * @var LanguageDao 
     */
    protected $languageDao;
    
    /**
     * @var LocaleDao 
     */
    protected $localeDao;
    

/// generovany konstruktor
    function __construct($appDir, $configDir, AdminModuleDao $adminModuleDao, AdminModuleSettingDao $adminModuleSettingDao, AdminRoleDao $adminRoleDao, AdminRoleModuleDao $adminRoleModuleDao, CountryDao $countryDao, LanguageDao $languageDao, LocaleDao $localeDao) {
        $this->appDir = $appDir;
        $this->configDir = $configDir;
        $this->adminModuleDao = $adminModuleDao;
        $this->adminModuleSettingDao = $adminModuleSettingDao;
        $this->adminRoleDao = $adminRoleDao;
        $this->adminRoleModuleDao = $adminRoleModuleDao;
        $this->countryDao = $countryDao;
        $this->languageDao = $languageDao;
        $this->localeDao = $localeDao;
    } 
    /**
     * vrati defaultni strankovac
     */
    public function getPaginatorConfig($maxPageSize) {
        $config = $this->adminModuleDao->createDefaultPaginatorConfig($maxPageSize);
        $config->clearOrders();
        $config->addOrder("moduleOrder");
        
        return $config;
    }
    
    /**
     * vrati aktivni moduly
     */
    public function getPage(PaginatorConfig $config) {
        return $this->adminModuleDao->paginate($config);
    }
    
    /**
     * @param integer $id
     * @return AdminModule
     */
    public function get($id) {
        $module = $this->adminModuleDao->load($id);
        $this->fillModule($module);
        
        return $module;
    }
    
    public function update(AdminModule $obj) {
        $this->adminModuleDao->update($obj);
        $this->saveModuleRoles($obj);
        
        foreach($obj->getModuleSettings() as $setting) {
            $this->adminModuleSettingDao->update($setting);
        }
    }

    public function delete($id) {
        $this->adminModuleDao->delete($id);
        $oldUserRoles = $this->adminRoleModuleDao->find(["fkAdminModuleId" => $id])->listResult();
        foreach($oldUserRoles as $oldUserRole) {
            $this->adminRoleModuleDao->delete($oldUserRole);
        }
        
        $settings = $this->adminModuleSettingDao->find(["fkAdminModuleId" => $id])->listResult();
        foreach($settings as $setting) {
            $this->adminModuleSettingDao->delete($setting);
        }
        
    }
    
    public function install(AdminModule $module) {
        $this->fillDefaultModuleSettings($module);
        
        $id = $this->adminModuleDao->save($module);
        
        foreach($module->getModuleSettings() as $adminSetting) {
            $adminSetting->setFkAdminModuleId($id);
            $this->adminModuleSettingDao->save($adminSetting);
        }
    }
    
    /**
     * @return array vrati typy vsech dostupnych modulu
     */
    public function getAvailableModulesTypes() {
        $moduleTypes = [];
        
        $installedModules = require $this->configDir ."/". Config::INSTALLED_MODULES_FILENAME;
        foreach($installedModules as $type => $value) {
            if( isset($value["path"]) ) { // ma cestu k modulu
                $moduleTypes[$type] = $type;
            }
        }
        asort($moduleTypes);
        
        return $moduleTypes;
    }
    
    public function fillDefaultModuleSettings(AdminModule $module) {
        $moduleTypes = [];
        
        $installedModules = require $this->configDir ."/". Config::INSTALLED_MODULES_FILENAME;
        
        if( !isset($installedModules[$module->getType()]) ) { // takovy module neni dostupny
            throw new Exception("Modul/Knihovna ". $module->getType() ." neni definovany v composer.json");
        }
        $moduleConfig = $installedModules[$module->getType()];
        if( !isset($moduleConfig["path"]) ) {
            throw new Exception($module->getType() ." není modul");
        }
        
        // sobir s nastavenimi
        $moduleSettingFile = $this->appDir .'/www/'. $moduleConfig["path"] ."/install/settings.php";
        if( file_exists($moduleSettingFile) ) { // nastaveni modulu existuje
            $settings = require_once $moduleSettingFile;
            
            // kopiruje nastaveni
            $moduleSettings = [];
            $module->setIcon($settings["icon"]);
            foreach($settings["moduleSettings"] as $paramIdentifier => $setting) {
                $adminSetting = new AdminModuleSetting();
                $adminSetting->setParamIdentifier($paramIdentifier);
                $adminSetting->setName($setting["name"]);
                $adminSetting->setValue($setting["value"]);
                
                $moduleSettings[] = $adminSetting;
            }
            $module->setModuleSettings($moduleSettings);
        }
    }

    protected function fillModule(AdminModule $adminModule) {
        $roles = $this->adminRoleDao->findModuleRoles($adminModule->getId())->listResult();
        $adminModule->setRoles($roles);
        
        $settings = $this->adminModuleSettingDao->find(["fkAdminModuleId" => $adminModule->getId()])->listResult();
        $adminModule->setModuleSettings($settings);
    }
    
    protected function saveModuleRoles(AdminModule $adminModule) {
        $oldUserRoles = $this->adminRoleModuleDao->find(["fkAdminModuleId" => $adminModule->getId()])->listResult();
        $oldUserRolesHash = [];
        foreach($oldUserRoles as $oldUserRole) {
            $oldUserRolesHash[$oldUserRole->getFkAdminRoleId()] = $oldUserRole;
        }
        
        foreach($adminModule->getRoles() as $role) {
            if( isset($oldUserRolesHash[$role->getId()]) ) {
                unset($oldUserRolesHash[$role->getId()]);
            } else {
                $moduleRole = new AdminRoleModule();
                $moduleRole->setFkAdminModuleId($adminModule->getId());
                $moduleRole->setFkAdminRoleId($role->getId());
                
                $this->adminRoleModuleDao->save($moduleRole);
            }
        }

        foreach($oldUserRolesHash as $oldUserRole) {
            $this->adminRoleModuleDao->delete($oldUserRole);
        }
    }
    
    
    
    public function getRoles() {
        return $this->adminRoleDao->find([], ["name" => OrderBy::ASC])->listResult();
    }

    public function getActiveCountries() {
        return $this->countryDao->find(["enabled" => true], ["name" => OrderBy::ASC])->listResult();
    }
    
    public function getActiveLanguages() {
        return $this->languageDao->find(["enabled" => true], ["name" => OrderBy::ASC])->listResult();
    }
    
    public function getActiveLocales() {
        return $this->localeDao->find([], ["localeTag" => OrderBy::ASC])->listResult();
    }
}